#!/usr/bin/env python3

"""
Game Fix Manager - A simple tool to fix common issues in the game
"""

import os
import json
import sys
import re
import shutil
from pathlib import Path

class GameFixManager:
    def __init__(self):
        self.project_root = Path(os.getcwd())
        self.unit_data = {}
        
        print(f"Game Fix Manager initialized at: {self.project_root}")
        
    def load_unit_data(self):
        """Load all unit data from JSON files"""
        print("\nLoading unit data...")
        
        data_path = self.project_root / "game" / "data" / "units"
        if not data_path.exists():
            print(f"Error: Unit data directory not found at {data_path}")
            return False
            
        count = 0
        for unit_file in data_path.glob("*.json"):
            try:
                with open(unit_file, 'r') as f:
                    faction = unit_file.stem
                    self.unit_data[faction] = json.load(f)
                    count += 1
                    print(f"  Loaded {faction} ({len(self.unit_data[faction])} unit types)")
            except Exception as e:
                print(f"  Error loading {unit_file}: {str(e)}")
                
        if count == 0:
            print("  No unit data files found.")
            return False
            
        return True
        
    def clean_pycache(self):
        """Clean all __pycache__ directories in the project"""
        print("\nCleaning __pycache__ directories...")
        
        pycache_dirs = list(self.project_root.glob("**/__pycache__"))
        if not pycache_dirs:
            print("  No __pycache__ directories found.")
            return
            
        print(f"  Found {len(pycache_dirs)} __pycache__ directories.")
        
        cleaned = 0
        for pycache_dir in pycache_dirs:
            try:
                shutil.rmtree(pycache_dir)
                print(f"  Removed: {pycache_dir}")
                cleaned += 1
            except Exception as e:
                print(f"  Error removing {pycache_dir}: {str(e)}")
                
        print(f"\n✅ Cleaned {cleaned} of {len(pycache_dirs)} __pycache__ directories.")
    
    def add_chaos_berzerkers_mission(self):
        """Add a mission featuring Khorne Berzerkers"""
        print("\nAdding Khorne Berzerkers mission...")
        
        mission_file = self.project_root / "game" / "ui" / "mission_scene.py"
        
        if not mission_file.exists():
            print(f"  Error: Mission file not found at {mission_file}")
            return
            
        # Read the current file
        try:
            with open(mission_file, 'r') as f:
                content = f.read()
        except Exception as e:
            print(f"  Error reading mission file: {str(e)}")
            return
            
        # Find the mission list array
        mission_list_pattern = re.compile(r'self\.mission_list\s*=\s*\[(.*?)\]', re.DOTALL)
        match = mission_list_pattern.search(content)
        
        if not match:
            print("  Error: Couldn't find mission list in the file.")
            return
            
        mission_list_text = match.group(1)
        
        # Look for the Khorne Berzerkers mission
        if "Khorne Berzerkers" in mission_list_text or "mission_khorne" in mission_list_text:
            print("  Khorne Berzerkers mission already exists.")
            return
            
        # Find a place to add the new mission
        last_mission_end = mission_list_text.rstrip().rfind("}")
        
        if last_mission_end == -1:
            print("  Error: Couldn't find a place to add the new mission.")
            return
            
        # Create a new mission for Khorne Berzerkers
        berzerker_mission = """
            ,{
                "id": "mission_khorne",
                "name": "Blood Crusade: Khorne Berzerkers",
                "description": "Face the fearsome Khorne Berzerkers in their blood-soaked fortress.",
                "difficulty": "Hard",
                "reward": 800,
                "location": "Chaos Fortress",
                "unlocked": True,
                "completed": False,
                "enemy_faction": "chaos_marines",
                "enemy_unit_types": ["chaos_berzerker", "chaos_marine", "chaos_marine"]
            }"""
            
        # Insert the new mission
        updated_mission_list = mission_list_text[:last_mission_end+1] + berzerker_mission + mission_list_text[last_mission_end+1:]
        updated_content = content.replace(mission_list_text, updated_mission_list)
        
        # Write back the updated content
        try:
            with open(mission_file, 'w') as f:
                f.write(updated_content)
            print("✅ Added Khorne Berzerkers mission.")
        except Exception as e:
            print(f"  Error updating mission file: {str(e)}")
    
    def add_lord_of_skulls_mission(self):
        """Add a mission featuring the Lord of Skulls"""
        print("\nAdding Lord of Skulls mission...")
        
        mission_file = self.project_root / "game" / "ui" / "mission_scene.py"
        
        if not mission_file.exists():
            print(f"  Error: Mission file not found at {mission_file}")
            return
            
        # Read the current file
        try:
            with open(mission_file, 'r') as f:
                content = f.read()
        except Exception as e:
            print(f"  Error reading mission file: {str(e)}")
            return
            
        # Find the mission list array
        mission_list_pattern = re.compile(r'self\.mission_list\s*=\s*\[(.*?)\]', re.DOTALL)
        match = mission_list_pattern.search(content)
        
        if not match:
            print("  Error: Couldn't find mission list in the file.")
            return
            
        mission_list_text = match.group(1)
        
        # Look for the Lord of Skulls mission
        if "Lord of Skulls" in mission_list_text or "mission_skulls" in mission_list_text:
            print("  Lord of Skulls mission already exists.")
            return
            
        # Find a place to add the new mission
        last_mission_end = mission_list_text.rstrip().rfind("}")
        
        if last_mission_end == -1:
            print("  Error: Couldn't find a place to add the new mission.")
            return
            
        # Create a new mission for the Lord of Skulls
        skulls_mission = """
            ,{
                "id": "mission_skulls",
                "name": "Blood for the Blood God",
                "description": "Battle the monstrous Khorne Lord of Skulls and its followers.",
                "difficulty": "Super Hard",
                "reward": 1200,
                "location": "Desecrated Imperial Cathedral",
                "unlocked": True,
                "completed": False,
                "enemy_faction": "chaos_marines",
                "enemy_unit_types": ["lord_of_skulls", "chaos_berzerker", "chaos_marine"]
            }"""
            
        # Insert the new mission
        updated_mission_list = mission_list_text[:last_mission_end+1] + skulls_mission + mission_list_text[last_mission_end+1:]
        updated_content = content.replace(mission_list_text, updated_mission_list)
        
        # Write back the updated content
        try:
            with open(mission_file, 'w') as f:
                f.write(updated_content)
            print("✅ Added Lord of Skulls mission.")
        except Exception as e:
            print(f"  Error updating mission file: {str(e)}")
    
    def add_ork_battle_wagon_mission(self):
        """Add a mission featuring the Ork Battle Wagon"""
        print("\nAdding Ork Battle Wagon mission...")
        
        mission_file = self.project_root / "game" / "ui" / "mission_scene.py"
        
        if not mission_file.exists():
            print(f"  Error: Mission file not found at {mission_file}")
            return
            
        # Read the current file
        try:
            with open(mission_file, 'r') as f:
                content = f.read()
        except Exception as e:
            print(f"  Error reading mission file: {str(e)}")
            return
            
        # Find the mission list array
        mission_list_pattern = re.compile(r'self\.mission_list\s*=\s*\[(.*?)\]', re.DOTALL)
        match = mission_list_pattern.search(content)
        
        if not match:
            print("  Error: Couldn't find mission list in the file.")
            return
            
        mission_list_text = match.group(1)
        
        # Look for the Battle Wagon mission
        if "Battle Wagon" in mission_list_text or "mission_wagon" in mission_list_text:
            print("  Battle Wagon mission already exists.")
            return
            
        # Find a place to add the new mission
        last_mission_end = mission_list_text.rstrip().rfind("}")
        
        if last_mission_end == -1:
            print("  Error: Couldn't find a place to add the new mission.")
            return
            
        # Create a new mission for the Battle Wagon
        wagon_mission = """
            ,{
                "id": "mission_wagon",
                "name": "Mek's Pride: Battle Wagon",
                "description": "Face a horde of Orks led by their prized Battle Wagon.",
                "difficulty": "Hard",
                "reward": 900,
                "location": "Ork Mek Workshop",
                "unlocked": True,
                "completed": False,
                "enemy_faction": "orks",
                "enemy_unit_types": ["battle_wagon", "nob", "loota"]
            }"""
            
        # Insert the new mission
        updated_mission_list = mission_list_text[:last_mission_end+1] + wagon_mission + mission_list_text[last_mission_end+1:]
        updated_content = content.replace(mission_list_text, updated_mission_list)
        
        # Write back the updated content
        try:
            with open(mission_file, 'w') as f:
                f.write(updated_content)
            print("✅ Added Battle Wagon mission.")
        except Exception as e:
            print(f"  Error updating mission file: {str(e)}")
    
    def add_battle_wagon_definition(self):
        """Add the Battle Wagon unit definition to the Orks file"""
        print("\nAdding Battle Wagon unit definition...")
        
        orks_file = self.project_root / "game" / "data" / "units" / "orks.json"
        
        if not orks_file.exists():
            print(f"  Error: Orks unit file not found at {orks_file}")
            return
            
        # Read the current file
        try:
            with open(orks_file, 'r') as f:
                orks_data = json.load(f)
        except Exception as e:
            print(f"  Error reading Orks file: {str(e)}")
            return
            
        # Check if Battle Wagon already exists
        if "battle_wagon" in orks_data:
            print("  Battle Wagon definition already exists.")
            return
            
        # Add the Battle Wagon definition
        orks_data["battle_wagon"] = {
            "name": "Ork Battle Wagon",
            "hp": 250,
            "max_hp": 250,
            "attack": 30,
            "defense": 20,
            "move_range": 3,
            "attack_range": 3,
            "abilities": ["big_shoota", "heavy_armor", "transport"],
            "size": [2, 2],
            "attack_damage_type": "normal",
            "faction": "orks"
        }
        
        # Write back the updated data
        try:
            with open(orks_file, 'w') as f:
                json.dump(orks_data, f, indent=2)
            print("✅ Added Battle Wagon definition to Orks.")
        except Exception as e:
            print(f"  Error updating Orks file: {str(e)}")
    
    def add_chaos_berzerker_definition(self):
        """Add the Khorne Berzerker unit definition to the Chaos Marines file"""
        print("\nAdding Khorne Berzerker unit definition...")
        
        chaos_file = self.project_root / "game" / "data" / "units" / "chaos_marines.json"
        
        if not chaos_file.exists():
            print(f"  Error: Chaos Marines unit file not found at {chaos_file}")
            return
            
        # Read the current file
        try:
            with open(chaos_file, 'r') as f:
                chaos_data = json.load(f)
        except Exception as e:
            print(f"  Error reading Chaos Marines file: {str(e)}")
            return
            
        # Check if Berzerker already exists
        if "chaos_berzerker" in chaos_data:
            print("  Khorne Berzerker definition already exists.")
            return
            
        # Add the Berzerker definition
        chaos_data["chaos_berzerker"] = {
            "name": "Khorne Berzerker",
            "hp": 140,
            "max_hp": 140,
            "attack": 35,
            "defense": 15,
            "move_range": 4,
            "attack_range": 1,
            "abilities": ["chain_axe", "chainblade", "blood_for_the_blood_god", "berserk_rage"],
            "size": [1, 1],
            "attack_damage_type": "normal",
            "faction": "chaos_marines"
        }
        
        # Write back the updated data
        try:
            with open(chaos_file, 'w') as f:
                json.dump(chaos_data, f, indent=2)
            print("✅ Added Khorne Berzerker definition to Chaos Marines.")
        except Exception as e:
            print(f"  Error updating Chaos Marines file: {str(e)}")

    def fix_battle_scene(self):
        """Add missing methods to the battle scene file"""
        print("\nFixing Battle Scene file...")
        
        battle_scene_path = self.project_root / "game" / "ui" / "battle_scene.py"
        
        if not battle_scene_path.exists():
            print(f"  Error: Battle Scene file not found at {battle_scene_path}")
            return
            
        try:
            with open(battle_scene_path, 'r') as f:
                content = f.read()
        except Exception as e:
            print(f"  Error reading Battle Scene file: {str(e)}")
            return
            
        # Check if _start_enemy_turn method is missing
        if "_start_enemy_turn" not in content:
            print("  Adding missing _start_enemy_turn method...")
            
            # Find where to add the method
            # Look for _end_enemy_turn as a reference point
            end_enemy_turn_pos = content.find("def _end_enemy_turn")
            
            if end_enemy_turn_pos != -1:
                # Add before _end_enemy_turn
                new_method = """
    def _start_enemy_turn(self):
        \"\"\"Start the enemy turn\"\"\"
        print("Starting enemy turn")
        self.turn = "enemy"
        self.current_phase = "enemy"
        
        # Schedule the AI to run after a short delay
        pygame.time.set_timer(pygame.USEREVENT+1, 500, 1)  # Set timer for 500ms
        pygame.event.post(pygame.event.Event(pygame.USEREVENT+1, {'action': 'run_enemy_ai'}))
        
"""
                content = content[:end_enemy_turn_pos] + new_method + content[end_enemy_turn_pos:]
            else:
                print("  Couldn't find a good place to add _start_enemy_turn method.")
                
        # Check if AI event handling is missing
        if "run_enemy_ai" not in content or "USEREVENT+1" not in content:
            print("  Adding enemy AI event handling...")
            
            # Find the handle_event method
            handle_event_pos = content.find("def handle_event(self, event):")
            
            if handle_event_pos != -1:
                # Find a good place within handle_event to add our code
                # Look for USEREVENT+2 or some other event handling
                event_handling_pos = content.find("pygame.USEREVENT", handle_event_pos)
                
                if event_handling_pos != -1:
                    # Find the start of this condition
                    line_start = content.rfind("if", handle_event_pos, event_handling_pos)
                    if line_start != -1:
                        event_handling_start = content.find("elif", line_start, event_handling_pos)
                        if event_handling_start != -1:
                            # Add our event handling code
                            new_event_handling = """
        # Handle run enemy AI event
        elif event.type == pygame.USEREVENT+1:
            if 'action' in event.dict and event.dict['action'] == 'run_enemy_ai':
                self._run_enemy_ai()
                # Clear the timer
                pygame.time.set_timer(pygame.USEREVENT+1, 0)
                
"""
                            content = content[:event_handling_start] + new_event_handling + content[event_handling_start:]
                        else:
                            print("  Couldn't find a good place to add AI event handling.")
                else:
                    print("  Couldn't find event handling section in handle_event method.")
            else:
                print("  Couldn't find handle_event method.")
                
        # Fix update method if it's automatically running enemy AI
        update_pos = content.find("def update(self):")
        if update_pos != -1 and "run_enemy_ai" in content[update_pos:update_pos+500]:
            print("  Fixing update method to use event-based AI...")
            
            # Find the end of the update method
            next_method_pos = content.find("def ", update_pos + 10)
            if next_method_pos != -1:
                # Replace with a fixed version
                new_update_method = """
    def update(self):
        \"\"\"Update the battle scene\"\"\"
        # No automatic enemy AI processing here - now handled by events
        pass
        
"""
                content = content[:update_pos] + new_update_method + content[next_method_pos:]
            else:
                print("  Couldn't find the end of the update method.")
                
        # Write back the updated content
        try:
            with open(battle_scene_path, 'w') as f:
                f.write(content)
            print("✅ Fixed Battle Scene file.")
        except Exception as e:
            print(f"  Error updating Battle Scene file: {str(e)}")
            
    def fix_unit_movement(self):
        """Modify battle scene to allow moving all units once per turn"""
        print("\nFixing unit movement system...")
        
        battle_scene_path = self.project_root / "game" / "ui" / "battle_scene.py"
        
        if not battle_scene_path.exists():
            print(f"  Error: Battle Scene file not found at {battle_scene_path}")
            return
            
        try:
            with open(battle_scene_path, 'r') as f:
                content = f.read()
        except Exception as e:
            print(f"  Error reading Battle Scene file: {str(e)}")
            return
            
        # Update the __init__ method to add moved_units tracking
        if "self.moved_units = set()" not in content:
            print("  Adding moved_units tracking...")
            
            # Find the place to add it - right after acted_units
            acted_units_pos = content.find("self.acted_units = set()")
            
            if acted_units_pos != -1:
                insert_pos = acted_units_pos + len("self.acted_units = set()") + 1
                new_code = "\n        # Track which units have moved this turn\n        self.moved_units = set()"
                content = content[:insert_pos] + new_code + content[insert_pos:]
            else:
                print("  Couldn't find a good place to add moved_units tracking.")
                return
                
        # Update the _handle_button_click method to separate movement from actions
        if "self.moved_units.add" not in content:
            print("  Updating move logic...")
            
            # Find the move button handler
            move_button_handler = content.find('if button_id == "move":')
            if move_button_handler != -1:
                # Find the check for already acted units
                acted_check = content.find("if self.selected_unit[\"id\"] in self.acted_units:", move_button_handler)
                
                if acted_check != -1:
                    # Modify to check moved_units instead of acted_units for movement
                    next_line_end = content.find("\n", acted_check) + 1
                    new_check = "if self.selected_unit[\"id\"] in self.moved_units:\n                    self.battle_message = f\"{self.selected_unit['name']} has already moved this turn.\"\n                    self.last_action = f\"Tried to move unit that already moved: {self.selected_unit['name']}\"\n                    return\n                "
                    content = content[:acted_check] + new_check + content[next_line_end:]
                else:
                    print("  Couldn't find acted_units check in move button handler.")
                    return
            else:
                print("  Couldn't find move button handler.")
                return
                
        # Update the _handle_grid_click method to track moved units separately
        if "self.moved_units.add" not in content:
            print("  Updating grid click handler for movement...")
            
            # Find where the unit is marked as having acted after moving
            acted_units_add = content.find("self.acted_units.add(self.selected_unit[\"id\"])", 0, content.find("def _handle_button_click"))
            
            if acted_units_add != -1:
                # Add moved_units tracking after the existing acted_units line
                line_end = content.find("\n", acted_units_add)
                insert_pos = line_end + 1
                new_code = "                    self.moved_units.add(self.selected_unit[\"id\"])  # Track that unit has moved\n"
                content = content[:insert_pos] + new_code + content[insert_pos:]
                
                # Modify the acted_units check to allow attacking after moving
                content = content.replace("unit_has_acted = self.selected_unit[\"id\"] in self.acted_units", 
                                         "unit_has_acted = self.selected_unit[\"id\"] in self.acted_units\n                unit_has_moved = self.selected_unit[\"id\"] in self.moved_units")
            else:
                print("  Couldn't find where acted_units is updated in grid click handler.")
                return
                
        # Update the _end_player_turn method to reset moved_units
        if "self.moved_units = set()" not in content:
            print("  Updating end turn handler...")
            
            end_turn_method = content.find("def _end_player_turn(self):")
            if end_turn_method != -1:
                reset_acted_units = content.find("self.acted_units = set()", end_turn_method)
                
                if reset_acted_units != -1:
                    line_end = content.find("\n", reset_acted_units)
                    insert_pos = line_end + 1
                    new_code = "        # Reset moved units for next turn\n        self.moved_units = set()\n"
                    content = content[:insert_pos] + new_code + content[insert_pos:]
                else:
                    print("  Couldn't find where acted_units is reset in end turn method.")
                    return
            else:
                print("  Couldn't find _end_player_turn method.")
                return
                
        # Write back the updated content
        try:
            with open(battle_scene_path, 'w') as f:
                f.write(content)
            print("✅ Fixed unit movement system to allow moving all units once per turn.")
        except Exception as e:
            print(f"  Error updating Battle Scene file: {str(e)}")
    
    def verify_game_system(self):
        """Verify that all game systems are working correctly"""
        print("\nVerifying game systems...")
        
        issues_found = 0
        fixes_applied = 0
        
        # 1. Check unit definitions
        print("\n- Checking unit definitions...")
        if not self.load_unit_data():
            print("  ❌ Error: Could not load unit data!")
            issues_found += 1
        else:
            # Check required factions
            required_factions = ["space_marines", "orks", "tau", "chaos_marines"]
            missing_factions = [f for f in required_factions if f not in self.unit_data]
            
            if missing_factions:
                print(f"  ❌ Error: Missing factions: {', '.join(missing_factions)}")
                issues_found += 1
            else:
                print("  ✓ All required factions found.")
                
            # Check required units
            required_units = {
                "space_marines": ["tactical", "devastator", "scout", "terminator", "captain", "razorback"],
                "orks": ["boy", "nob", "shoota", "loota", "warboss", "battle_wagon"],
                "tau": ["tau_fire_warrior", "battle_suit", "devilfish"],
                "chaos_marines": ["chaos_marine", "havoc", "cultist", "chaos_berzerker", "lord_of_skulls"]
            }
            
            for faction, units in required_units.items():
                if faction in self.unit_data:
                    missing_units = [u for u in units if u not in self.unit_data[faction]]
                    if missing_units:
                        print(f"  ❌ Error: Missing {faction} units: {', '.join(missing_units)}")
                        issues_found += 1
                        
                        # Auto-fix missing units
                        if "battle_wagon" in missing_units and faction == "orks":
                            self.add_battle_wagon_definition()
                            fixes_applied += 1
                            
                        if "chaos_berzerker" in missing_units and faction == "chaos_marines":
                            self.add_chaos_berzerker_definition()
                            fixes_applied += 1
                            
        # 2. Check battle scene
        print("\n- Checking battle scene implementation...")
        battle_scene_path = self.project_root / "game" / "ui" / "battle_scene.py"
        
        if not battle_scene_path.exists():
            print(f"  ❌ Error: Battle scene file not found at {battle_scene_path}")
            issues_found += 1
        else:
            try:
                with open(battle_scene_path, 'r') as f:
                    content = f.read()
                    
                # Check for critical methods
                required_methods = [
                    "_handle_grid_click", 
                    "_handle_button_click", 
                    "_calculate_move_range",
                    "_calculate_attack_range",
                    "_run_enemy_ai",
                    "_try_move_enemy_unit",
                    "_end_player_turn"
                ]
                
                missing_methods = []
                for method in required_methods:
                    if f"def {method}" not in content:
                        missing_methods.append(method)
                
                if missing_methods:
                    print(f"  ❌ Error: Missing critical methods in battle scene: {', '.join(missing_methods)}")
                    issues_found += 1
                else:
                    print("  ✓ All critical battle scene methods present.")
                
                # Check for _start_enemy_turn method
                if "def _start_enemy_turn" not in content:
                    print("  ❌ Error: Missing _start_enemy_turn method")
                    issues_found += 1
                    self.fix_battle_scene()
                    fixes_applied += 1
                else:
                    print("  ✓ Enemy turn handling is present.")
                    
                # Check for moved_units tracking
                if "self.moved_units = set()" not in content:
                    print("  ❌ Error: Unit movement system needs update")
                    issues_found += 1
                    self.fix_unit_movement()
                    fixes_applied += 1
                else:
                    print("  ✓ Unit movement tracking is present.")
                    
                # Check for invalid color argument in button rendering
                button_color_none = "bg_color=(40, 40, 60) if unit_has_acted else None"
                if button_color_none in content:
                    print("  ❌ Error: Invalid button color argument (None) found")
                    issues_found += 1
                    
                    # Fix the button color issue
                    fixed_content = content.replace(
                        "bg_color=(40, 40, 60) if unit_has_acted else None",
                        "bg_color=(40, 40, 60) if unit_has_acted else (50, 50, 50)"
                    )
                    fixed_content = fixed_content.replace(
                        "ability_bg_color = None", 
                        "ability_bg_color = (50, 50, 50)"
                    )
                    fixed_content = fixed_content.replace(
                        "bg_color=ability_bg_color if not ability_active else None",
                        "bg_color=ability_bg_color"
                    )
                    
                    try:
                        with open(battle_scene_path, 'w') as f:
                            f.write(fixed_content)
                        print("  ✓ Fixed button color arguments.")
                        fixes_applied += 1
                    except Exception as e:
                        print(f"  ❌ Error fixing button color: {str(e)}")
                else:
                    print("  ✓ Button rendering looks good.")
                    
            except Exception as e:
                print(f"  ❌ Error checking battle scene: {str(e)}")
                issues_found += 1
                
        # 3. Check mission definitions
        print("\n- Checking mission definitions...")
        mission_scene_path = self.project_root / "game" / "ui" / "mission_scene.py"
        
        if not mission_scene_path.exists():
            print(f"  ❌ Error: Mission scene file not found at {mission_scene_path}")
            issues_found += 1
        else:
            try:
                with open(mission_scene_path, 'r') as f:
                    content = f.read()
                
                # Check for mission list
                if "self.mission_list" not in content:
                    print("  ❌ Error: No mission list found in mission scene")
                    issues_found += 1
                else:
                    print("  ✓ Mission list found.")
                    
                    # Check for specific missions
                    missing_missions = []
                    if "mission_khorne" not in content:
                        missing_missions.append("Khorne Berzerkers mission")
                        self.add_chaos_berzerkers_mission()
                        fixes_applied += 1
                        
                    if "mission_skulls" not in content:
                        missing_missions.append("Lord of Skulls mission")
                        self.add_lord_of_skulls_mission()
                        fixes_applied += 1
                        
                    if "mission_wagon" not in content:
                        missing_missions.append("Battle Wagon mission")
                        self.add_ork_battle_wagon_mission()
                        fixes_applied += 1
                        
                    if missing_missions:
                        print(f"  ❌ Missing missions: {', '.join(missing_missions)} (auto-fixed)")
                    else:
                        print("  ✓ All special missions are present.")
                    
            except Exception as e:
                print(f"  ❌ Error checking mission scene: {str(e)}")
                issues_found += 1
                
        # 4. Check AI system
        print("\n- Checking AI system...")
        ai_path = self.project_root / "game" / "ai"
        
        if not ai_path.exists() or not ai_path.is_dir():
            print("  ❌ Error: AI directory not found")
            issues_found += 1
        else:
            # Check for essential AI files
            required_ai_files = [
                "ai_manager.py", 
                "ork_ai.py", 
                "tau_ai.py", 
                "chaos_ai.py"
            ]
            
            missing_files = []
            for file in required_ai_files:
                if not (ai_path / file).exists():
                    missing_files.append(file)
                    
            if missing_files:
                print(f"  ❌ Error: Missing AI files: {', '.join(missing_files)}")
                issues_found += 1
            else:
                print("  ✓ All required AI files are present.")
                
            # Check for AI manager
            ai_manager_path = ai_path / "ai_manager.py"
            if ai_manager_path.exists():
                try:
                    with open(ai_manager_path, 'r') as f:
                        content = f.read()
                        
                    # Check all AI factions are registered
                    required_factions = ["orks", "tau", "chaos_marines"]
                    missing_factions = []
                    
                    for faction in required_factions:
                        if f"'{faction}'" not in content and f'"{faction}"' not in content:
                            missing_factions.append(faction)
                            
                    if missing_factions:
                        print(f"  ❌ Error: AI Manager missing faction registrations: {', '.join(missing_factions)}")
                        issues_found += 1
                    else:
                        print("  ✓ AI Manager has all required factions.")
                        
                except Exception as e:
                    print(f"  ❌ Error checking AI manager: {str(e)}")
                    issues_found += 1
                
        # Summary
        print("\n=== Verification Summary ===")
        if issues_found == 0:
            print("✅ All systems verified. No issues found!")
        else:
            print(f"Found {issues_found} issues, applied {fixes_applied} automatic fixes.")
            if issues_found > fixes_applied:
                print(f"⚠️ {issues_found - fixes_applied} issues could not be automatically fixed and require manual attention.")
            else:
                print("✅ All detected issues have been fixed automatically!")
                
        return issues_found == 0
        
    def show_menu(self):
        """Show the main menu of the fix manager"""
        while True:
            print("\n" + "="*50)
            print("GAME FIX MANAGER - MAIN MENU")
            print("="*50)
            print("1. Clean __pycache__ directories")
            print("2. Add Khorne Berzerkers mission")
            print("3. Add Lord of Skulls mission")
            print("4. Add Battle Wagon mission")
            print("5. Add Battle Wagon unit definition")
            print("6. Add Khorne Berzerker unit definition")
            print("7. Fix Battle Scene file")
            print("8. Fix unit movement system")
            print("9. Verify all game systems")
            print("10. Fix ALL issues (run all fixes)")
            print("11. Exit")
            
            choice = input("\nEnter your choice (1-11): ")
            
            if choice == '1':
                self.clean_pycache()
            elif choice == '2':
                self.add_chaos_berzerkers_mission()
            elif choice == '3':
                self.add_lord_of_skulls_mission()
            elif choice == '4':
                self.add_ork_battle_wagon_mission()
            elif choice == '5':
                self.add_battle_wagon_definition()
            elif choice == '6':
                self.add_chaos_berzerker_definition()
            elif choice == '7':
                self.fix_battle_scene()
            elif choice == '8':
                self.fix_unit_movement()
            elif choice == '9':
                self.verify_game_system()
            elif choice == '10':
                print("\nRunning all fixes...")
                self.clean_pycache()
                self.add_chaos_berzerker_definition()
                self.add_chaos_berzerkers_mission()
                self.add_lord_of_skulls_mission()
                self.add_battle_wagon_definition()
                self.add_ork_battle_wagon_mission()
                self.fix_battle_scene()
                self.fix_unit_movement()
                # Verify everything after applying fixes
                print("\nVerifying all fixes were applied correctly...")
                self.verify_game_system()
                print("\n✅ All fixes completed!")
            elif choice == '11':
                print("\nExiting Fix Manager. Goodbye!")
                break
            else:
                print("\nInvalid choice. Please enter a number between 1 and 11.")

def main():
    """Main entry point for the fix manager"""
    try:
        manager = GameFixManager()
        manager.show_menu()
    except KeyboardInterrupt:
        print("\n\nFix Manager was interrupted. Exiting.")
    except Exception as e:
        print(f"\nAn error occurred: {str(e)}")
        
if __name__ == "__main__":
    main() 