#!/usr/bin/env python3
"""
Warhammer 40K: Heroes of the Imperium
Main game entry point
"""
import os
import sys
import pygame
from game.core.game_manager import GameManager

def main():
    """Main function to initialize and run the game"""
    # Initialize pygame
    pygame.init()
    pygame.display.set_caption("Warhammer 40K: Heroes of the Imperium")
    
    # Create screen (16:9 ratio)
    screen_width = 1280
    screen_height = 720
    screen = pygame.display.set_mode((screen_width, screen_height), pygame.RESIZABLE)
    
    # Set up clock for managing frame rate
    clock = pygame.time.Clock()
    fps = 60
    
    try:
        # Initialize game manager
        game_manager = GameManager(screen)
        
        # Main game loop
        running = True
        while running:
            # Process events
            for event in pygame.event.get():
                if event.type == pygame.QUIT:
                    running = False
                elif event.type == pygame.VIDEORESIZE:
                    # Handle window resize events
                    if not game_manager.settings.get("fullscreen", False):
                        # Only resize if not in fullscreen mode
                        screen = pygame.display.set_mode((event.w, event.h), pygame.RESIZABLE)
                        # Update game manager's screen reference
                        game_manager.screen = screen
                        # Notify scenes of resize
                        current_scene = game_manager.scenes[game_manager.current_state]
                        if hasattr(current_scene, 'on_resize'):
                            current_scene.on_resize(screen)
                # Pass events to game manager
                game_manager.handle_event(event)
            
            # Update game state
            game_manager.update()
            
            # Render the game
            screen.fill((0, 0, 0))  # Clear screen
            game_manager.render()
            pygame.display.flip()
            
            # Cap the frame rate
            clock.tick(fps)
            
    except Exception as e:
        print(f"Error: {e}")
        import traceback
        traceback.print_exc()
    finally:
        # Clean up pygame
        pygame.quit()
        
if __name__ == "__main__":
    main() 