"""
Terms and Conditions Scene
Displays the Game Detection terms and conditions to the user
"""
import pygame
from game.ui.scene import Scene
from game.core.game_detector import GameDetector

class TermsScene(Scene):
    """Scene for displaying and accepting terms and conditions"""
    
    def __init__(self, game_manager):
        """Initialize the terms scene"""
        super().__init__(game_manager)
        
        # Create game detector
        self.game_detector = GameDetector(game_manager)
        
        # Get terms and conditions
        self.terms_data = self.game_detector.get_terms_and_conditions()
        
        # UI state
        self.current_section = 0
        self.accepted_items = set()
        self.all_items = []
        self.scrollable_sections = True
        self.scroll_offset = 0
        self.max_scroll_offset = 0
        self.hover_button = None
        
        # Load saved scroll position
        self.section_scroll_offsets = [0] * len(self.terms_data["terms"])
        
        # Count total terms
        self._count_terms()
        
        # UI elements
        self.buttons = []
        self.checkboxes = []
        
        # Initialize UI elements
        self._initialize_buttons()
    
    def _count_terms(self):
        """Count the total number of term items"""
        self.total_items = 0
        for section in self.terms_data["terms"]:
            self.total_items += len(section["items"])
            # Build list of all items
            for item in section["items"]:
                self.all_items.append((section["id"], item))
    
    def _initialize_buttons(self):
        """Initialize buttons and UI elements"""
        # Clear existing
        self.buttons = []
        self.checkboxes = []
        
        screen_width = self.game_manager.screen.get_width()
        screen_height = self.game_manager.screen.get_height()
        
        # Navigation buttons
        button_width = 150
        button_height = 40
        button_spacing = 20
        
        # Previous section button
        if self.current_section > 0:
            prev_button = {
                "id": "prev_section",
                "text": "Previous",
                "rect": pygame.Rect(
                    20, 
                    screen_height - button_height - 20,
                    button_width,
                    button_height
                ),
                "color": (70, 70, 100),
                "hover_color": (90, 90, 120),
                "text_color": (255, 255, 255)
            }
            self.buttons.append(prev_button)
        
        # Next section button
        if self.current_section < len(self.terms_data["terms"]) - 1:
            next_button = {
                "id": "next_section",
                "text": "Next",
                "rect": pygame.Rect(
                    screen_width - button_width - 20, 
                    screen_height - button_height - 20,
                    button_width,
                    button_height
                ),
                "color": (70, 70, 100),
                "hover_color": (90, 90, 120),
                "text_color": (255, 255, 255)
            }
            self.buttons.append(next_button)
        
        # Accept all button
        if self.current_section == len(self.terms_data["terms"]) - 1:
            # Only show on last section
            accept_button = {
                "id": "accept_terms",
                "text": "Accept All Terms",
                "rect": pygame.Rect(
                    (screen_width - button_width*2) // 2,
                    screen_height - button_height - 20,
                    button_width*2,
                    button_height
                ),
                "color": (50, 100, 50),
                "hover_color": (70, 120, 70),
                "text_color": (255, 255, 255)
            }
            self.buttons.append(accept_button)
        
        # Decline button
        decline_button = {
            "id": "decline_terms",
            "text": "Decline",
            "rect": pygame.Rect(
                20, 
                20,
                100,
                30
            ),
            "color": (100, 50, 50),
            "hover_color": (120, 70, 70),
            "text_color": (255, 255, 255)
        }
        self.buttons.append(decline_button)
        
        # Accept All Section Items button
        accept_section_button = {
            "id": "accept_section",
            "text": "Accept All in Section",
            "rect": pygame.Rect(
                screen_width - 200, 
                20,
                180,
                30
            ),
            "color": (50, 100, 50),
            "hover_color": (70, 120, 70),
            "text_color": (255, 255, 255)
        }
        self.buttons.append(accept_section_button)
        
        # Select All checkbox (new)
        select_all_checkbox = {
            "id": "select_all",
            "text": "Select All Terms",
            "rect": pygame.Rect(
                (screen_width - 200) // 2,
                20,
                24,
                24
            ),
            "color": (50, 100, 50),
            "hover_color": (70, 120, 70),
            "text_color": (255, 255, 255),
            "checked": self._are_all_terms_checked()
        }
        self.buttons.append(select_all_checkbox)
        
        # Section checkboxes
        if self.current_section < len(self.terms_data["terms"]):
            section = self.terms_data["terms"][self.current_section]
            checkbox_y = 140
            checkbox_size = 24
            content_width = screen_width - 100
            
            for i, item in enumerate(section["items"]):
                # Ensure item starts with checkbox or add it
                if not item.startswith("☐"):
                    item = "☐ " + item
                
                # Create checkbox
                checkbox = {
                    "id": f"checkbox_{self.current_section}_{i}",
                    "section_id": section["id"],
                    "item_text": item,
                    "rect": pygame.Rect(
                        40,
                        checkbox_y + i * 40 - self.section_scroll_offsets[self.current_section],
                        checkbox_size,
                        checkbox_size
                    ),
                    "checked": self._is_item_checked(section["id"], item)
                }
                self.checkboxes.append(checkbox)
            
            # Set max scroll offset based on content height
            self.max_scroll_offset = max(0, (len(section["items"]) * 40) - (screen_height - 220))
    
    def _is_item_checked(self, section_id, item_text):
        """Check if an item is already accepted"""
        item_key = f"{section_id}:{item_text}"
        return item_key in self.accepted_items
    
    def _toggle_checkbox(self, checkbox_id):
        """Toggle checkbox state"""
        for checkbox in self.checkboxes:
            if checkbox["id"] == checkbox_id:
                item_key = f"{checkbox['section_id']}:{checkbox['item_text']}"
                
                if checkbox["checked"]:
                    # Uncheck
                    checkbox["checked"] = False
                    if item_key in self.accepted_items:
                        self.accepted_items.remove(item_key)
                else:
                    # Check
                    checkbox["checked"] = True
                    self.accepted_items.add(item_key)
                break
    
    def _accept_all_in_section(self):
        """Accept all items in current section"""
        section = self.terms_data["terms"][self.current_section]
        
        for i, item in enumerate(section["items"]):
            if not item.startswith("☐"):
                item = "☐ " + item
                
            item_key = f"{section['id']}:{item}"
            self.accepted_items.add(item_key)
            
            # Update checkbox visual
            for checkbox in self.checkboxes:
                if checkbox["section_id"] == section["id"]:
                    checkbox["checked"] = True
    
    def _accept_all_terms(self):
        """Accept all terms and enable the game detector"""
        # Add any missing items
        for section in self.terms_data["terms"]:
            for item in section["items"]:
                if not item.startswith("☐"):
                    item = "☐ " + item
                    
                item_key = f"{section['id']}:{item}"
                self.accepted_items.add(item_key)
        
        # Enable game detector
        settings = {
            "scan_settings": {
                "scan_steam": True,
                "scan_epic": True,
                "upload_data": True,
                "scan_frequency": "once"
            },
            "privacy_settings": {
                "anonymize_data": True,
                "store_locally_only": True,
                "allow_content_suggestions": True
            }
        }
        
        self.game_detector.accept_terms_and_conditions(True, settings)
        
        # Return to previous scene
        self.game_manager.change_state(self.game_manager.previous_state)
    
    def _decline_terms(self):
        """Decline terms and return to previous scene"""
        self.game_detector.accept_terms_and_conditions(False)
        self.game_manager.change_state(self.game_manager.previous_state)
    
    def _go_to_previous_section(self):
        """Go to previous section"""
        if self.current_section > 0:
            self.current_section -= 1
            self._initialize_buttons()
    
    def _go_to_next_section(self):
        """Go to next section"""
        if self.current_section < len(self.terms_data["terms"]) - 1:
            self.current_section += 1
            self._initialize_buttons()
    
    def enter(self):
        """Called when entering the scene"""
        self.hover_button = None
    
    def handle_event(self, event):
        """Handle pygame events"""
        if event.type == pygame.MOUSEMOTION:
            # Handle button hover
            mouse_pos = pygame.mouse.get_pos()
            self.hover_button = None
            
            for button in self.buttons:
                if self.is_point_inside_rect(mouse_pos, button["rect"]):
                    self.hover_button = button["id"]
                    break
        
        elif event.type == pygame.MOUSEBUTTONDOWN:
            if event.button == 1:  # Left click
                mouse_pos = pygame.mouse.get_pos()
                
                # Check buttons
                for button in self.buttons:
                    if self.is_point_inside_rect(mouse_pos, button["rect"]):
                        if button["id"] == "prev_section":
                            self._go_to_previous_section()
                        elif button["id"] == "next_section":
                            self._go_to_next_section()
                        elif button["id"] == "accept_terms":
                            self._accept_all_terms()
                        elif button["id"] == "decline_terms":
                            self._decline_terms()
                        elif button["id"] == "accept_section":
                            self._accept_all_in_section()
                        elif button["id"] == "select_all":
                            self._select_all_terms()
                        break
                
                # Check checkboxes
                for checkbox in self.checkboxes:
                    if self.is_point_inside_rect(mouse_pos, checkbox["rect"]):
                        self._toggle_checkbox(checkbox["id"])
                        break
            
            elif event.button == 4:  # Mouse wheel up
                # Scroll up
                self.section_scroll_offsets[self.current_section] = max(
                    0, 
                    self.section_scroll_offsets[self.current_section] - 20
                )
                self._initialize_buttons()
            
            elif event.button == 5:  # Mouse wheel down
                # Scroll down
                self.section_scroll_offsets[self.current_section] = min(
                    self.max_scroll_offset, 
                    self.section_scroll_offsets[self.current_section] + 20
                )
                self._initialize_buttons()
        
        elif event.type == pygame.KEYDOWN:
            if event.key == pygame.K_ESCAPE:
                self._decline_terms()
            elif event.key == pygame.K_SPACE and self.current_section < len(self.terms_data["terms"]) - 1:
                self._go_to_next_section()
    
    def update(self):
        """Update the scene"""
        pass
    
    def render(self, screen):
        """Render the scene"""
        # Fill the background
        screen.fill((30, 30, 40))
        
        # Get screen dimensions
        screen_width = screen.get_width()
        screen_height = screen.get_height()
        
        # Draw title
        title_font = pygame.font.SysFont(None, 48)
        title_text = "Game Detection Terms and Conditions"
        title_surface = title_font.render(title_text, True, (220, 220, 220))
        title_rect = title_surface.get_rect(center=(screen_width // 2, 60))
        screen.blit(title_surface, title_rect)
        
        # Draw section title if available
        if self.current_section < len(self.terms_data["terms"]):
            section = self.terms_data["terms"][self.current_section]
            
            section_font = pygame.font.SysFont(None, 36)
            section_text = f"Section {self.current_section + 1}: {section['section']}"
            section_surface = section_font.render(section_text, True, (200, 200, 200))
            section_rect = section_surface.get_rect(center=(screen_width // 2, 100))
            screen.blit(section_surface, section_rect)
            
            # Draw progress indicator
            progress_text = f"Section {self.current_section + 1} of {len(self.terms_data['terms'])}"
            progress_font = pygame.font.SysFont(None, 20)
            progress_surface = progress_font.render(progress_text, True, (150, 150, 150))
            screen.blit(progress_surface, (screen_width // 2 - progress_surface.get_width() // 2, 130))
            
            # Draw items with checkboxes
            checkbox_font = pygame.font.SysFont(None, 24)
            content_width = screen_width - 100
            
            # Draw each checkbox
            for checkbox in self.checkboxes:
                # Draw checkbox
                checkbox_color = (100, 200, 100) if checkbox["checked"] else (100, 100, 100)
                pygame.draw.rect(screen, checkbox_color, checkbox["rect"])
                pygame.draw.rect(screen, (200, 200, 200), checkbox["rect"], 1)
                
                if checkbox["checked"]:
                    # Draw checkmark
                    checkmark_points = [
                        (checkbox["rect"].left + 5, checkbox["rect"].top + 12),
                        (checkbox["rect"].left + 10, checkbox["rect"].top + 18),
                        (checkbox["rect"].left + 19, checkbox["rect"].top + 5)
                    ]
                    pygame.draw.lines(screen, (255, 255, 255), False, checkmark_points, 2)
                
                # Draw item text
                # Replace checkbox marker if it exists
                text = checkbox["item_text"].replace("☐ ", "")
                text_color = (220, 220, 220) if checkbox["checked"] else (200, 200, 200)
                
                # Word wrap text
                words = text.split(' ')
                line = ""
                lines = []
                line_y = checkbox["rect"].top
                for word in words:
                    test_line = line + word + " "
                    test_surface = checkbox_font.render(test_line, True, text_color)
                    if test_surface.get_width() > content_width - 80:
                        lines.append(line)
                        line = word + " "
                    else:
                        line = test_line
                if line:
                    lines.append(line)
                
                # Render each line
                for i, line in enumerate(lines):
                    line_surface = checkbox_font.render(line, True, text_color)
                    screen.blit(line_surface, (checkbox["rect"].right + 10, line_y + i * 24))
        
        # Draw buttons
        for button in self.buttons:
            if button["id"] == "select_all":
                # Draw checkbox style for select all
                checkbox_color = (100, 200, 100) if button.get("checked", False) else (100, 100, 100)
                pygame.draw.rect(screen, checkbox_color, button["rect"])
                pygame.draw.rect(screen, (200, 200, 200), button["rect"], 1)
                
                if button.get("checked", False):
                    # Draw checkmark
                    checkmark_points = [
                        (button["rect"].left + 5, button["rect"].top + 12),
                        (button["rect"].left + 10, button["rect"].top + 18),
                        (button["rect"].left + 19, button["rect"].top + 5)
                    ]
                    pygame.draw.lines(screen, (255, 255, 255), False, checkmark_points, 2)
                
                # Draw label text next to checkbox
                font = pygame.font.SysFont(None, 24)
                text_surface = font.render("Select All Terms", True, (220, 220, 220))
                screen.blit(text_surface, (button["rect"].right + 10, button["rect"].top + 4))
            else:
                # Regular button rendering
                color = button["hover_color"] if self.hover_button == button["id"] else button["color"]
                pygame.draw.rect(screen, color, button["rect"])
                pygame.draw.rect(screen, (150, 150, 150), button["rect"], 1)
                
                # Draw button text
                font = pygame.font.SysFont(None, 24)
                text_surface = font.render(button["text"], True, button["text_color"])
                text_rect = text_surface.get_rect(center=button["rect"].center)
                screen.blit(text_surface, text_rect)
        
        # Draw accepted terms count
        accepted_count = len(self.accepted_items)
        total_count = self.total_items
        accepted_text = f"Accepted Terms: {accepted_count}/{total_count}"
        accepted_font = pygame.font.SysFont(None, 20)
        accepted_surface = accepted_font.render(accepted_text, True, (150, 200, 150))
        screen.blit(accepted_surface, (20, screen_height - 60))
    
    def _are_all_terms_checked(self):
        """Check if all terms are accepted"""
        # Count all terms
        all_terms_count = 0
        for section in self.terms_data["terms"]:
            all_terms_count += len(section["items"])
        
        # If we've accepted the same number of terms as exist, all are checked
        return len(self.accepted_items) == all_terms_count
    
    def _select_all_terms(self):
        """Select all terms across all sections"""
        # Add all terms to accepted items
        for section in self.terms_data["terms"]:
            for item in section["items"]:
                if not item.startswith("☐"):
                    item = "☐ " + item
                
                item_key = f"{section['id']}:{item}"
                self.accepted_items.add(item_key)
        
        # Update current section's checkboxes
        for checkbox in self.checkboxes:
            checkbox["checked"] = True
            
        # Update the select all button
        for button in self.buttons:
            if button["id"] == "select_all":
                button["checked"] = True
                break
    
    def on_resize(self, screen):
        """Handle screen resize event"""
        # Re-initialize all UI elements with the new screen dimensions
        self._initialize_buttons()
        
        # Adjust scroll offsets if they've become invalid with the new screen size
        for i, offset in enumerate(self.section_scroll_offsets):
            if i < len(self.terms_data["terms"]):
                section = self.terms_data["terms"][i]
                screen_height = screen.get_height()
                max_offset = max(0, (len(section["items"]) * 40) - (screen_height - 220))
                
                # Clamp the offset to valid range
                self.section_scroll_offsets[i] = min(offset, max_offset)
        
        # Update max scroll offset for current section
        if self.current_section < len(self.terms_data["terms"]):
            section = self.terms_data["terms"][self.current_section]
            screen_height = screen.get_height()
            self.max_scroll_offset = max(0, (len(section["items"]) * 40) - (screen_height - 220)) 