"""
Base Scene Module
Provides a template for all game scenes
"""
import pygame

class Scene:
    """
    Base scene class for all game scenes
    """
    def __init__(self, game_manager):
        """Initialize the scene"""
        self.game_manager = game_manager
        self.resource_manager = game_manager.resource_manager
    
    def enter(self):
        """Called when entering this scene"""
        pass
    
    def exit(self):
        """Called when exiting this scene"""
        pass
    
    def handle_event(self, event):
        """Handle pygame events"""
        pass
    
    def update(self):
        """Update scene logic"""
        pass
    
    def render(self, screen):
        """Render the scene"""
        pass
    
    def on_resize(self, screen):
        """Called when the screen changes size or display mode"""
        # Base implementation just updates the reference if needed
        # Subclasses should override to reposition UI elements
        pass
    
    def draw_text(self, screen, text, position, font_name="medium", color=(255, 255, 255), 
                 centered=False):
        """Helper method to draw text on the screen"""
        font = self.resource_manager.get_font(font_name)
        text_surface = font.render(text, True, color)
        
        if centered:
            text_rect = text_surface.get_rect(center=position)
        else:
            text_rect = text_surface.get_rect(topleft=position)
            
        screen.blit(text_surface, text_rect)
        return text_rect
    
    def draw_button(self, screen, text, position, size, font_name="medium", 
                   color=(200, 200, 200), hover_color=(255, 255, 255), 
                   bg_color=(50, 50, 50), hover_bg_color=(70, 70, 70),
                   centered=True, hover=False, border_radius=5):
        """Helper method to draw a button"""
        # Create button rectangle
        if centered:
            rect = pygame.Rect(0, 0, size[0], size[1])
            rect.center = position
        else:
            rect = pygame.Rect(position[0], position[1], size[0], size[1])
        
        # Draw button background
        bg = hover_bg_color if hover else bg_color
        pygame.draw.rect(screen, bg, rect, border_radius=border_radius)
        
        # Draw button text
        text_color = hover_color if hover else color
        font = self.resource_manager.get_font(font_name)
        text_surface = font.render(text, True, text_color)
        text_rect = text_surface.get_rect(center=rect.center)
        screen.blit(text_surface, text_rect)
        
        return rect
    
    def is_point_inside_rect(self, point, rect):
        """Check if a point is inside a rectangle"""
        return rect.collidepoint(point) 