"""
Save/Load Scene Module
Handles the UI for saving and loading games
"""
import pygame
import os
import time
from game.ui.scene import Scene
from game.core.game_state import GameState

class SaveLoadScene(Scene):
    """
    Scene for saving and loading games
    """
    def __init__(self, game_manager):
        """Initialize the save/load scene"""
        super().__init__(game_manager)
        self.save_manager = game_manager.save_manager
        self.mode = "load"  # "save" or "load"
        self.selected_slot = 0
        self.hovering_slot = None
        self.screenshot = None
        self.buttons = []
        self.confirmation_dialog = None
        self.message = None
        self.message_timer = 0
        
        # Initialize UI elements
        self._initialize_ui()
        
    def enter(self):
        """Called when entering the scene"""
        # Refresh save slots
        self.save_manager._refresh_save_slots()
        
        # Take a screenshot for previews when saving
        if self.mode == "save":
            self._take_screenshot()
            
        # Reset hovering state
        self.hovering_slot = None
        self.confirmation_dialog = None
        self.message = None
        self.message_timer = 0
        
        # Initialize UI
        self._initialize_ui()
    
    def _take_screenshot(self):
        """Take a screenshot of the current game state"""
        try:
            # Get the current screen
            screen = self.game_manager.screen
            
            # Create a copy of the screen for the screenshot
            self.screenshot = screen.copy()
            
            # Darken the screenshot (for previews)
            dark = pygame.Surface(self.screenshot.get_size(), flags=pygame.SRCALPHA)
            dark.fill((0, 0, 0, 100))  # Semi-transparent black
            self.screenshot.blit(dark, (0, 0))
        except Exception as e:
            print(f"Error taking screenshot: {e}")
            self.screenshot = None
    
    def _initialize_ui(self):
        """Initialize UI elements"""
        self.buttons = []
        
        screen_width = self.game_manager.screen.get_width()
        screen_height = self.game_manager.screen.get_height()
        
        # Title and mode buttons
        title_y = 40
        mode_button_width = 150
        mode_button_height = 40
        mode_button_spacing = 20
        
        # Save button
        self.buttons.append({
            "id": "mode_save",
            "text": "Save Game",
            "rect": pygame.Rect(
                screen_width // 2 - mode_button_width - mode_button_spacing // 2,
                title_y + 50,
                mode_button_width,
                mode_button_height
            ),
            "color": (50, 100, 50) if self.mode == "save" else (70, 70, 90),
            "hover_color": (70, 120, 70) if self.mode == "save" else (90, 90, 110),
            "text_color": (255, 255, 255),
            "action": lambda: self.set_mode("save")
        })
        
        # Load button
        self.buttons.append({
            "id": "mode_load",
            "text": "Load Game",
            "rect": pygame.Rect(
                screen_width // 2 + mode_button_spacing // 2,
                title_y + 50,
                mode_button_width,
                mode_button_height
            ),
            "color": (50, 50, 100) if self.mode == "load" else (70, 70, 90),
            "hover_color": (70, 70, 120) if self.mode == "load" else (90, 90, 110),
            "text_color": (255, 255, 255),
            "action": lambda: self.set_mode("load")
        })
        
        # Back button
        self.buttons.append({
            "id": "back",
            "text": "Back",
            "rect": pygame.Rect(
                20,
                screen_height - 60,
                100,
                40
            ),
            "color": (100, 50, 50),
            "hover_color": (120, 70, 70),
            "text_color": (255, 255, 255),
            "action": self._go_back
        })
        
        # Action buttons
        if self.mode == "save":
            # Save button
            self.buttons.append({
                "id": "save",
                "text": "Save Game",
                "rect": pygame.Rect(
                    screen_width - 150 - 20,
                    screen_height - 60,
                    150,
                    40
                ),
                "color": (50, 100, 50),
                "hover_color": (70, 120, 70),
                "text_color": (255, 255, 255),
                "action": self._save_to_selected_slot
            })
        elif self.mode == "load":
            # Load button
            self.buttons.append({
                "id": "load",
                "text": "Load Game",
                "rect": pygame.Rect(
                    screen_width - 150 - 20,
                    screen_height - 60,
                    150,
                    40
                ),
                "color": (50, 50, 100),
                "hover_color": (70, 70, 120),
                "text_color": (255, 255, 255),
                "action": self._load_from_selected_slot
            })
            
            # Delete button
            self.buttons.append({
                "id": "delete",
                "text": "Delete Save",
                "rect": pygame.Rect(
                    screen_width - 330 - 20,
                    screen_height - 60,
                    150,
                    40
                ),
                "color": (100, 50, 50),
                "hover_color": (120, 70, 70),
                "text_color": (255, 255, 255),
                "action": self._confirm_delete_save
            })
    
    def set_mode(self, mode):
        """Set the mode (save or load)"""
        if mode != self.mode:
            self.mode = mode
            self.hovering_slot = None
            self.confirmation_dialog = None
            
            if mode == "save":
                self._take_screenshot()
                
            self._initialize_ui()
    
    def handle_event(self, event):
        """Handle pygame events"""
        if self.confirmation_dialog:
            # Handle confirmation dialog events
            if event.type == pygame.MOUSEBUTTONDOWN:
                if event.button == 1:  # Left click
                    mouse_pos = pygame.mouse.get_pos()
                    
                    # Check yes button
                    if self.is_point_inside_rect(mouse_pos, self.confirmation_dialog["yes_rect"]):
                        self.confirmation_dialog["yes_action"]()
                        self.confirmation_dialog = None
                        return
                    
                    # Check no button
                    if self.is_point_inside_rect(mouse_pos, self.confirmation_dialog["no_rect"]):
                        self.confirmation_dialog = None
                        return
            
            # Don't process other events when dialog is open
            return
            
        if event.type == pygame.MOUSEMOTION:
            mouse_pos = pygame.mouse.get_pos()
            
            # Check save slot hover
            self.hovering_slot = None
            for i, slot_info in enumerate(self._get_save_slot_rects()):
                if self.is_point_inside_rect(mouse_pos, slot_info["rect"]):
                    self.hovering_slot = i
                    break
            
            # Check button hover
            for button in self.buttons:
                button["hover"] = self.is_point_inside_rect(mouse_pos, button["rect"])
            
        elif event.type == pygame.MOUSEBUTTONDOWN:
            if event.button == 1:  # Left click
                mouse_pos = pygame.mouse.get_pos()
                
                # Check save slot click
                for i, slot_info in enumerate(self._get_save_slot_rects()):
                    if self.is_point_inside_rect(mouse_pos, slot_info["rect"]):
                        self.selected_slot = i
                        if self.mode == "save":
                            self._save_to_selected_slot()
                        return
                
                # Check button click
                for button in self.buttons:
                    if self.is_point_inside_rect(mouse_pos, button["rect"]):
                        button["action"]()
                        return
        
        elif event.type == pygame.KEYDOWN:
            if event.key == pygame.K_ESCAPE:
                self._go_back()
    
    def update(self):
        """Update the scene"""
        # Update message timer
        if self.message and self.message_timer > 0:
            self.message_timer -= 1
            if self.message_timer <= 0:
                self.message = None
    
    def render(self, screen):
        """Render the scene"""
        # Fill the background
        screen.fill((30, 30, 40))
        
        # Draw title
        screen_width = screen.get_width()
        screen_height = screen.get_height()
        
        title_text = "Save Game" if self.mode == "save" else "Load Game"
        title_pos = (screen_width // 2, 40)
        title_font = pygame.font.SysFont(None, 48)
        title_surface = title_font.render(title_text, True, (220, 220, 220))
        title_rect = title_surface.get_rect(center=title_pos)
        screen.blit(title_surface, title_rect)
        
        # Draw mode buttons
        for button in self.buttons:
            if button["id"].startswith("mode_"):
                color = button["hover_color"] if button.get("hover", False) else button["color"]
                pygame.draw.rect(screen, color, button["rect"])
                pygame.draw.rect(screen, (100, 100, 100), button["rect"], 1)
                
                text_surface = pygame.font.SysFont(None, 24).render(button["text"], True, button["text_color"])
                text_rect = text_surface.get_rect(center=button["rect"].center)
                screen.blit(text_surface, text_rect)
        
        # Draw save slots
        self._draw_save_slots(screen)
        
        # Draw action buttons
        for button in self.buttons:
            if not button["id"].startswith("mode_"):
                color = button["hover_color"] if button.get("hover", False) else button["color"]
                pygame.draw.rect(screen, color, button["rect"])
                pygame.draw.rect(screen, (100, 100, 100), button["rect"], 1)
                
                text_surface = pygame.font.SysFont(None, 24).render(button["text"], True, button["text_color"])
                text_rect = text_surface.get_rect(center=button["rect"].center)
                screen.blit(text_surface, text_rect)
        
        # Draw message if active
        if self.message:
            self._draw_message(screen)
            
        # Draw confirmation dialog if active
        if self.confirmation_dialog:
            self._draw_confirmation_dialog(screen)
    
    def _draw_save_slots(self, screen):
        """Draw save slots on the screen"""
        slots = self._get_save_slot_rects()
        
        for i, slot_info in enumerate(slots):
            rect = slot_info["rect"]
            
            # Determine slot color
            if i == self.selected_slot:
                # Selected slot
                color = (60, 90, 120)
                border_color = (120, 180, 240)
                border_width = 2
            elif i == self.hovering_slot:
                # Hovering slot
                color = (50, 70, 100)
                border_color = (100, 150, 200)
                border_width = 2
            else:
                # Normal slot
                color = (40, 40, 60)
                border_color = (80, 80, 100)
                border_width = 1
            
            # Draw slot background
            pygame.draw.rect(screen, color, rect)
            pygame.draw.rect(screen, border_color, rect, border_width)
            
            # Draw slot number
            slot_num_text = f"Slot {i + 1}"
            slot_num_surface = pygame.font.SysFont(None, 18).render(slot_num_text, True, (180, 180, 180))
            screen.blit(slot_num_surface, (rect.x + 10, rect.y + 10))
            
            # Get save data for this slot if it exists
            save_data = None
            for slot in self.save_manager.save_slots:
                if slot["slot"] == i:
                    save_data = slot
                    break
            
            if save_data:
                # Draw save metadata
                metadata = save_data["metadata"]
                
                # Draw save info
                save_date = metadata["save_date"]
                date_surface = pygame.font.SysFont(None, 18).render(save_date, True, (180, 180, 180))
                screen.blit(date_surface, (rect.x + 10, rect.y + 30))
                
                player_name = metadata["player_name"]
                name_surface = pygame.font.SysFont(None, 22).render(player_name, True, (220, 220, 220))
                screen.blit(name_surface, (rect.x + 10, rect.y + 50))
                
                level_text = f"Level {metadata['player_level']}"
                level_surface = pygame.font.SysFont(None, 20).render(level_text, True, (200, 200, 160))
                screen.blit(level_surface, (rect.x + 10, rect.y + 75))
                
                missions_text = f"Missions: {metadata['missions_completed']}"
                missions_surface = pygame.font.SysFont(None, 18).render(missions_text, True, (180, 180, 180))
                screen.blit(missions_surface, (rect.x + 10, rect.y + 95))
                
                # Draw screenshot thumbnail if available
                if metadata["screenshot"] and os.path.exists(metadata["screenshot"]):
                    try:
                        thumb = pygame.image.load(metadata["screenshot"])
                        thumb = pygame.transform.scale(thumb, (120, 70))
                        screen.blit(thumb, (rect.right - 130, rect.y + 20))
                    except:
                        pass
            else:
                # Draw empty slot text
                if self.mode == "save":
                    empty_text = "Click to save"
                else:
                    empty_text = "Empty slot"
                    
                empty_surface = pygame.font.SysFont(None, 24).render(empty_text, True, (150, 150, 150))
                empty_rect = empty_surface.get_rect(center=(rect.centerx, rect.centery))
                screen.blit(empty_surface, empty_rect)
                
                # If saving, show screenshot preview
                if self.mode == "save" and self.screenshot and i == self.selected_slot:
                    thumb = pygame.transform.scale(self.screenshot, (120, 70))
                    screen.blit(thumb, (rect.right - 130, rect.y + 20))
    
    def _get_save_slot_rects(self):
        """Get rectangles for all save slots"""
        screen_width = self.game_manager.screen.get_width()
        screen_height = self.game_manager.screen.get_height()
        
        # Calculate slot dimensions
        slot_width = screen_width - 100
        slot_height = 120
        slot_spacing = 20
        start_y = 120
        
        slots = []
        for i in range(self.save_manager.MAX_SAVE_SLOTS):
            rect = pygame.Rect(
                50,
                start_y + (slot_height + slot_spacing) * i,
                slot_width,
                slot_height
            )
            
            slots.append({
                "slot": i,
                "rect": rect
            })
        
        return slots
    
    def _draw_confirmation_dialog(self, screen):
        """Draw a confirmation dialog"""
        screen_width = screen.get_width()
        screen_height = screen.get_height()
        
        # Draw darkened background
        dark = pygame.Surface((screen_width, screen_height), flags=pygame.SRCALPHA)
        dark.fill((0, 0, 0, 180))  # Semi-transparent black
        screen.blit(dark, (0, 0))
        
        # Dialog dimensions
        dialog_width = 400
        dialog_height = 200
        dialog_x = (screen_width - dialog_width) // 2
        dialog_y = (screen_height - dialog_height) // 2
        
        # Draw dialog box
        dialog_rect = pygame.Rect(dialog_x, dialog_y, dialog_width, dialog_height)
        pygame.draw.rect(screen, (50, 50, 70), dialog_rect)
        pygame.draw.rect(screen, (100, 100, 120), dialog_rect, 2)
        
        # Draw title
        title_surface = pygame.font.SysFont(None, 30).render(self.confirmation_dialog["title"], True, (220, 220, 220))
        title_rect = title_surface.get_rect(center=(dialog_x + dialog_width // 2, dialog_y + 40))
        screen.blit(title_surface, title_rect)
        
        # Draw message
        message_surface = pygame.font.SysFont(None, 24).render(self.confirmation_dialog["message"], True, (200, 200, 200))
        message_rect = message_surface.get_rect(center=(dialog_x + dialog_width // 2, dialog_y + 80))
        screen.blit(message_surface, message_rect)
        
        # Draw buttons
        yes_rect = pygame.Rect(dialog_x + 80, dialog_y + dialog_height - 60, 100, 40)
        no_rect = pygame.Rect(dialog_x + dialog_width - 180, dialog_y + dialog_height - 60, 100, 40)
        
        # Store button rects for interaction
        self.confirmation_dialog["yes_rect"] = yes_rect
        self.confirmation_dialog["no_rect"] = no_rect
        
        # Yes button
        pygame.draw.rect(screen, (50, 100, 50), yes_rect)
        pygame.draw.rect(screen, (70, 120, 70), yes_rect, 2)
        yes_surface = pygame.font.SysFont(None, 24).render("Yes", True, (220, 220, 220))
        yes_text_rect = yes_surface.get_rect(center=yes_rect.center)
        screen.blit(yes_surface, yes_text_rect)
        
        # No button
        pygame.draw.rect(screen, (100, 50, 50), no_rect)
        pygame.draw.rect(screen, (120, 70, 70), no_rect, 2)
        no_surface = pygame.font.SysFont(None, 24).render("No", True, (220, 220, 220))
        no_text_rect = no_surface.get_rect(center=no_rect.center)
        screen.blit(no_surface, no_text_rect)
    
    def _draw_message(self, screen):
        """Draw a temporary message"""
        screen_width = screen.get_width()
        
        # Message dimensions
        message_height = 40
        padding = 20
        message_width = len(self.message) * 10 + padding * 2  # Rough estimate
        message_x = (screen_width - message_width) // 2
        message_y = 100
        
        # Draw message background
        message_rect = pygame.Rect(message_x, message_y, message_width, message_height)
        if "error" in self.message.lower():
            # Error message
            pygame.draw.rect(screen, (100, 30, 30), message_rect)
            pygame.draw.rect(screen, (150, 50, 50), message_rect, 2)
        else:
            # Success message
            pygame.draw.rect(screen, (30, 100, 30), message_rect)
            pygame.draw.rect(screen, (50, 150, 50), message_rect, 2)
        
        # Draw message text
        message_surface = pygame.font.SysFont(None, 24).render(self.message, True, (220, 220, 220))
        message_text_rect = message_surface.get_rect(center=message_rect.center)
        screen.blit(message_surface, message_text_rect)
    
    def _go_back(self):
        """Go back to the previous screen"""
        self.game_manager.change_state(self.game_manager.previous_state)
    
    def _save_to_selected_slot(self):
        """Save the game to the selected slot"""
        success = self.save_manager.save_game(self.selected_slot, self.screenshot)
        
        if success:
            self.message = "Game saved successfully!"
            self.message_timer = 90  # Show for 90 frames (about 1.5 seconds)
        else:
            self.message = "Error saving game"
            self.message_timer = 120  # Show for 2 seconds
        
        # Refresh UI
        self._initialize_ui()
    
    def _load_from_selected_slot(self):
        """Load the game from the selected slot"""
        # Check if there's a save in this slot
        save_exists = False
        for slot in self.save_manager.save_slots:
            if slot["slot"] == self.selected_slot:
                save_exists = True
                break
        
        if not save_exists:
            self.message = "No save in this slot"
            self.message_timer = 90
            return
        
        # Confirm loading
        self.confirmation_dialog = {
            "title": "Load Game",
            "message": "Are you sure you want to load this game? Unsaved progress will be lost.",
            "yes_action": self._confirm_load_game,
            "no_action": lambda: None
        }
    
    def _confirm_load_game(self):
        """Confirm and load the game"""
        success = self.save_manager.load_game(self.selected_slot)
        
        if success:
            # Return to main menu after loading
            self.game_manager.change_state(GameState.MAIN_MENU)
        else:
            self.message = "Error loading game"
            self.message_timer = 120
    
    def _confirm_delete_save(self):
        """Confirm deletion of a save"""
        # Check if there's a save in this slot
        save_exists = False
        for slot in self.save_manager.save_slots:
            if slot["slot"] == self.selected_slot:
                save_exists = True
                break
        
        if not save_exists:
            self.message = "No save in this slot"
            self.message_timer = 90
            return
        
        # Show confirmation dialog
        self.confirmation_dialog = {
            "title": "Delete Save",
            "message": "Are you sure you want to delete this save? This cannot be undone.",
            "yes_action": self._confirm_delete,
            "no_action": lambda: None
        }
    
    def _confirm_delete(self):
        """Confirm and delete the save"""
        success = self.save_manager.delete_save(self.selected_slot)
        
        if success:
            self.message = "Save deleted successfully"
            self.message_timer = 90
        else:
            self.message = "Error deleting save"
            self.message_timer = 120
        
        # Refresh UI
        self._initialize_ui() 