"""
Mission Selection Scene Module
"""
import pygame
from game.ui.scene import Scene
from game.core.game_state import GameState

class MissionScene(Scene):
    """
    Mission selection scene
    Displays available missions for the player
    """
    def __init__(self, game_manager):
        """Initialize the mission scene"""
        super().__init__(game_manager)
        self.buttons = []
        self.hover_button = None
        self.mission_list = []
        self.scroll_offset = 0
        self.selected_mission = None
        
        # --- Filter additions ---
        self.available_factions = ["All"] # Start with All
        self.selected_faction_filter = "All" # Default filter
        self.faction_filter_buttons = [] # To store clickable rects
        
        # --- ADD Font Initialization ---
        self.fonts = {
            "small": pygame.font.Font(None, 24),
            "medium": pygame.font.Font(None, 32),
            "large": pygame.font.Font(None, 48)
        }
        # -----------------------------
        
        # Background color - dark military green (Warhammer 40K theme)
        self.bg_color = (20, 30, 20)
        
        # Message display
        self.show_message = False
        self.message_text = ""
        self.message_color = (255, 255, 255)
        self.message_timer = 0
        self.message_duration = 3.0  # seconds
        
        # Initialize sample missions (will be loaded from data files later)
        self._initialize_sample_missions()
        self._populate_factions() # Populate factions after loading list
        
    def _initialize_sample_missions(self):
        """Initialize the list of available missions - ALL UNLOCKED"""
        # Now only lists unit type IDs
        self.mission_list = [
            {
                "id": "tutorial",
                "name": "Training Grounds",
                "description": "Basic training mission to learn the fundamentals of combat.",
                "difficulty": "Easy",
                "reward": 100,
                "location": "Training Grounds",
                "unlocked": True,
                "completed": False,
                "enemy_faction": "orks",
                "enemy_unit_types": ["boy", "boy"]
            },
            {
                "id": "mission_1",
                "name": "Border Worlds: Scout the Perimeter",
                "description": "Scout the border worlds for signs of Chaos activity.",
                "difficulty": "Easy",
                "reward": 200,
                "location": "Border Worlds",
                "unlocked": True,
                "completed": False,
                "enemy_faction": "orks",
                "enemy_unit_types": ["boy", "nob", "shoota"]
            },
            {
                "id": "mission_2",
                "name": "Abandoned Temple: Chaos Cult",
                "description": "Investigate reports of Chaos cult activity in an abandoned temple.",
                "difficulty": "Medium",
                "reward": 300,
                "location": "Abandoned Temple",
                "unlocked": True,
                "completed": False,
                "enemy_faction": "chaos_marines",
                "enemy_unit_types": ["chaos_marine", "chaos_marine", "defiler"]
            },
            {
                "id": "mission_3",
                "name": "Jungle Outpost: Ork Waaagh!",
                "description": "Stop an Ork Waaagh! before it can gather more forces.",
                "difficulty": "Medium",
                "reward": 400,
                "location": "Jungle Outpost",
                "unlocked": True,
                "completed": False,
                "enemy_faction": "orks",
                "enemy_unit_types": ["nob", "loota", "warboss"]
            },
            {
                "id": "mission_4",
                "name": "Imperial Fortress: Chaos Invasion",
                "description": "Defend an Imperial fortress from a Chaos invasion.",
                "difficulty": "Hard",
                "reward": 500,
                "location": "Imperial Fortress",
                "unlocked": True,
                "completed": False,
                "enemy_faction": "chaos_marines",
                "enemy_unit_types": ["terminator_chaos", "defiler", "havoc"]
            },
            {
                "id": "mission_5",
                "name": "Space Port: Chaos Ambush",
                "description": "Escape from a Chaos ambush at a critical space port.",
                "difficulty": "Hard",
                "reward": 600,
                "location": "Space Port",
                "unlocked": True,
                "completed": False,
                "enemy_faction": "chaos_marines",
                "enemy_unit_types": ["chaos_marine", "chaos_marine", "terminator_chaos"]
            },
            {
                "id": "mission_6",
                "name": "Ork Stronghold: Final Assault",
                "description": "Launch a final assault on the Ork stronghold.",
                "difficulty": "Hard",
                "reward": 700,
                "location": "Ork Stronghold",
                "unlocked": True,
                "completed": False,
                "enemy_faction": "orks",
                "enemy_unit_types": ["nob", "warboss", "loota"]
            },
            {
                "id": "mission_7",
                "name": "Tau Enclave: Recon in Force",
                "description": "Gather intelligence on Tau forces in a nearby enclave.",
                "difficulty": "Medium",
                "reward": 450,
                "location": "Tau Enclave",
                "unlocked": True,
                "completed": False,
                "enemy_faction": "tau",
                "enemy_unit_types": ["tau_fire_warrior", "tau_fire_warrior", "battle_suit"]
            },
            {
                "id": "mission_8",
                "name": "Desert Outpost: Necron Awakening",
                "description": "Investigate strange energy readings indicating Necron activity.",
                "difficulty": "Medium",
                "reward": 500,
                "location": "Desert Outpost",
                "unlocked": True,
                "completed": False,
                "enemy_faction": "necrons",
                "enemy_unit_types": ["necron_warrior", "necron_warrior", "necron_scarab"]
            },
            {
                "id": "mission_9",
                "name": "Hive City: Purge the Genestealer Cult",
                "description": "Root out a Genestealer Cult infestation in the lower hive.",
                "difficulty": "Hard",
                "reward": 650,
                "location": "Hive City Undercroft",
                "unlocked": True,
                "completed": False,
                "enemy_faction": "tyranids",
                "enemy_unit_types": ["genestealer", "hybrid_acolyte", "genestealer"]
            },
            {
                "id": "mission_10",
                "name": "Forge World: Tech-Heresy",
                "description": "Investigate rumors of tech-heresy on a nearby Forge World.",
                "difficulty": "Hard",
                "reward": 750,
                "location": "Forge World Sigma-7",
                "unlocked": True,
                "completed": False,
                "enemy_faction": "chaos_marines",
                "enemy_unit_types": ["chaos_marine", "havoc", "sorcerer"]
            },
            {
                "id": "mission_11",
                "name": "Asteroid Field: Eldar Pirates",
                "description": "Engage Eldar Corsair pirates operating from an asteroid base.",
                "difficulty": "Medium",
                "reward": 550,
                "location": "Asteroid Belt Delta",
                "unlocked": True,
                "completed": False,
                "enemy_faction": "eldar",
                "enemy_unit_types": ["eldar_guardian", "eldar_striking_scorpion", "eldar_guardian"]
            },
            {
                "id": "mission_12",
                "name": "Tau Outpost: Hammerhead Assault",
                "description": "Neutralize a Tau outpost that has been firing on Imperial supply lines.",
                "difficulty": "Hard",
                "reward": 700,
                "location": "Tau Frontier Outpost",
                "unlocked": True,
                "completed": False,
                "enemy_faction": "tau",
                "enemy_unit_types": ["tau_fire_warrior", "battle_suit", "tau_fire_warrior"]
            },
            {
                "id": "mission_13",
                "name": "Tau Stronghold: Devil Fish Fleet",
                "description": "Assault a heavily fortified Tau stronghold guarded by Devil Fish transports.",
                "difficulty": "Super Hard",
                "reward": 1000,
                "location": "Tau Command Center",
                "unlocked": True,
                "completed": False,
                "enemy_faction": "tau",
                "enemy_unit_types": ["devilfish", "battle_suit", "devilfish"]
            },
            {
                "id": "mission_14",
                "name": "Chaos Incursion: Warp Strike",
                "description": "Repel a Chaos Space Marine warband that has launched a surprise attack on Imperial territory.",
                "difficulty": "Hard",
                "reward": 800,
                "location": "Imperial Outpost Alpha",
                "unlocked": True,
                "completed": False,
                "enemy_faction": "chaos_marines",
                "enemy_unit_types": ["chaos_marine", "havoc", "cultist"]
            },
            {
                "id": "mission_15",
                "name": "Chaos Stronghold: Sorcerer's Lair",
                "description": "Assault the stronghold of a powerful Chaos Sorcerer and his elite Terminator guard.",
                "difficulty": "Super Hard",
                "reward": 1200,
                "location": "Corrupted Temple",
                "unlocked": True,
                "completed": False,
                "enemy_faction": "chaos_marines",
                "enemy_unit_types": ["sorcerer", "terminator_chaos", "defiler"]
            },
            {
                "id": "mission_16",
                "name": "Blood Crusade: Lord of Skulls",
                "description": "Face the ultimate challenge - a Khorne Lord of Skulls daemon engine leading a blood crusade against Imperial forces.",
                "difficulty": "Super Hard",
                "reward": 1500,
                "location": "Blood-Soaked Plains",
                "unlocked": True,
                "completed": False,
                "enemy_faction": "chaos_marines",
                "enemy_unit_types": ["lord_of_skulls", "terminator_chaos", "havoc"]
            },
            {
                "id": "mission_17",
                "name": "Eldar Craftworld: Webway Portal",
                "description": "Secure an ancient Webway portal being used by Eldar forces to raid Imperial worlds.",
                "difficulty": "Hard",
                "reward": 800,
                "location": "Webway Nexus",
                "unlocked": True,
                "completed": False,
                "enemy_faction": "eldar",
                "enemy_unit_types": ["eldar_dire_avenger", "eldar_warlock", "eldar_guardian"]
            },
            {
                "id": "mission_18",
                "name": "Eldar Shrine: Farseer's Vision",
                "description": "Confront a powerful Eldar Farseer who has foreseen your arrival and prepared an ambush.",
                "difficulty": "Hard",
                "reward": 900,
                "location": "Ancient Eldar Shrine",
                "unlocked": True,
                "completed": False,
                "enemy_faction": "eldar",
                "enemy_unit_types": ["eldar_farseer", "eldar_dire_avenger", "eldar_howling_banshee"]
            },
            {
                "id": "mission_19",
                "name": "Eldar Stronghold: Wraith Constructs",
                "description": "Infiltrate an Eldar stronghold protected by deadly Wraithlords and spirit constructs.",
                "difficulty": "Super Hard",
                "reward": 1100,
                "location": "Eldar Outpost Isha",
                "unlocked": True,
                "completed": False,
                "enemy_faction": "eldar",
                "enemy_unit_types": ["eldar_wraithlord", "eldar_warlock", "eldar_fire_dragon"]
            }
        ]
        
    def _populate_factions(self):
        """Dynamically find unique factions from the mission list."""
        factions = set() 
        for mission in self.mission_list:
            if "enemy_faction" in mission:
                factions.add(mission["enemy_faction"])
        # Create sorted list starting with "All"
        self.available_factions = ["All"] + sorted(list(factions))
        # Reset filter when populating (optional, could preserve)
        self.selected_faction_filter = "All"

    def _get_filtered_missions(self):
        """Return the list of missions matching the current faction filter."""
        if self.selected_faction_filter == "All":
            return self.mission_list
        else:
            return [m for m in self.mission_list if m.get("enemy_faction") == self.selected_faction_filter]

    def enter(self):
        """Called when entering the mission scene"""
        self.hover_button = None
        # Reset filter and repopulate factions in case mission list changes elsewhere
        # (or if unlocking/completion affects available factions in future)
        self._populate_factions() 
        self.scroll_offset = 0 # Reset scroll on entering/filtering
        
        # Check if returning from battle
        if hasattr(self.game_manager, 'last_battle_result') and self.game_manager.last_battle_result:
            if self.game_manager.last_battle_result == 'victory':
                self.show_message = True
                self.message_text = "Mission Completed Successfully!"
                self.message_color = (100, 255, 100)  # Green
                
                # Mark the last selected mission as completed
                for mission in self.mission_list:
                    if mission["id"] == self.selected_mission:
                        mission["completed"] = True
                        break # Stop searching once completed mission is found
            
            elif self.game_manager.last_battle_result == 'defeat':
                self.show_message = True
                self.message_text = "Mission Failed. Try again."
                self.message_color = (255, 100, 100)  # Red
            
            # Reset the battle result
            self.game_manager.last_battle_result = None
            self.message_timer = 0
        else:
            # Reset selection if entering normally (not from battle)
            self.selected_mission = None 
        
    def handle_event(self, event):
        """Handle pygame events"""
        if event.type == pygame.MOUSEMOTION:
            mouse_pos = pygame.mouse.get_pos()
            self.hover_button = None
            # Check mission buttons
            for button_info in self.buttons:
                if self.is_point_inside_rect(mouse_pos, button_info["rect"]):
                    self.hover_button = button_info["id"]
                    break
            # Check faction filter buttons
            if not self.hover_button:
                 for button_info in self.faction_filter_buttons:
                     if self.is_point_inside_rect(mouse_pos, button_info["rect"]):
                         self.hover_button = button_info["id"]
                         break
        
        elif event.type == pygame.MOUSEBUTTONDOWN:
            if event.button == 1:  # Left click
                mouse_pos = pygame.mouse.get_pos()
                clicked_handled = False
                # Check mission buttons first
                for button_info in self.buttons:
                    if self.is_point_inside_rect(mouse_pos, button_info["rect"]):
                        self._handle_button_click(button_info["id"])
                        clicked_handled = True; break
                # Check faction filters if no mission button clicked
                if not clicked_handled:
                     for button_info in self.faction_filter_buttons:
                        if self.is_point_inside_rect(mouse_pos, button_info["rect"]):
                            self._handle_button_click(button_info["id"])
                            clicked_handled = True; break
            
            # Handle scrolling
            elif event.button == 4:  # Scroll up
                self.scroll_offset = max(0, self.scroll_offset - 1)
            elif event.button == 5:  # Scroll down
                filtered_list = self._get_filtered_missions()
                # Adjust max_scroll based on filtered list length
                max_scroll = max(0, len(filtered_list) - 4) # Assuming 4 visible missions
                self.scroll_offset = min(max_scroll, self.scroll_offset + 1)
        
        elif event.type == pygame.KEYDOWN:
            if event.key == pygame.K_SPACE or event.key == pygame.K_RETURN:
                if self.hover_button:
                    self._handle_button_click(self.hover_button)
            elif event.key == pygame.K_ESCAPE:
                # Return to the main menu
                self.game_manager.change_state(GameState.MAIN_MENU)
            elif event.key == pygame.K_UP:
                self.scroll_offset = max(0, self.scroll_offset - 1)
            elif event.key == pygame.K_DOWN:
                filtered_list = self._get_filtered_missions()
                max_scroll = max(0, len(filtered_list) - 4)
                self.scroll_offset = min(max_scroll, self.scroll_offset + 1)
                
    def update(self):
        """Update the mission scene"""
        # Update message timer
        if self.show_message:
            self.message_timer += 0.016  # Approximately 60 FPS
            if self.message_timer >= self.message_duration:
                self.show_message = False
        
        # Ensure scroll offset is valid if list shrinks due to filtering
        filtered_list = self._get_filtered_missions()
        max_scroll = max(0, len(filtered_list) - 4)
        self.scroll_offset = min(self.scroll_offset, max_scroll)
        
    def render(self, screen):
        """Render the mission scene"""
        # Fill the background
        screen.fill(self.bg_color)
        
        screen_width, screen_height = screen.get_size()
        
        # Draw the header
        header_text = "MISSION SELECT"
        header_pos = (screen_width // 2, 50)
        self.draw_text(screen, header_text, header_pos, "large", (200, 200, 200), True)
        
        # Draw faction filters
        self.faction_filter_buttons = [] # Reset clickable areas
        filter_start_x = 50
        filter_y = 70 # Position below header
        filter_spacing = 20
        for faction_name in self.available_factions:
            is_selected = (faction_name == self.selected_faction_filter)
            is_hovered = (self.hover_button == f"filter_{faction_name}")
            
            text_color = (255, 255, 0) if is_selected else (200, 200, 200)
            if is_hovered and not is_selected: 
                 text_color = (255, 255, 255) # Brighter on hover
            
            # Use a slightly larger font for filters
            text_surface = self.fonts["medium"].render(faction_name.upper(), True, text_color)
            text_rect = text_surface.get_rect(topleft=(filter_start_x, filter_y))
            
            # Add underline if selected or hovered
            if is_selected or is_hovered:
                 underline_rect = pygame.Rect(text_rect.left, text_rect.bottom + 1, text_rect.width, 2)
                 pygame.draw.rect(screen, text_color, underline_rect)
                 
            screen.blit(text_surface, text_rect)
            
            # Store clickable area
            button_id = f"filter_{faction_name}"
            self.faction_filter_buttons.append({"id": button_id, "rect": text_rect})
            
            filter_start_x = text_rect.right + filter_spacing
            
        # Draw mission list
        mission_panel_rect = pygame.Rect(50, 100, screen_width - 100, screen_height - 200)
        pygame.draw.rect(screen, (40, 50, 40), mission_panel_rect, border_radius=5)
        
        # Draw filtered mission list
        self.buttons = [] # Reset mission buttons
        filtered_missions = self._get_filtered_missions()
        
        # Adjust scroll offset if it's now invalid after filtering
        max_scroll_filtered = max(0, len(filtered_missions) - 4)
        self.scroll_offset = min(self.scroll_offset, max_scroll_filtered)
        
        visible_missions = filtered_missions[self.scroll_offset:self.scroll_offset + 4]
        
        if not visible_missions and filtered_missions:
             # If scroll offset is too high after filtering, reset it
             self.scroll_offset = 0
             visible_missions = filtered_missions[self.scroll_offset:self.scroll_offset + 4]
             
        mission_start_y = mission_panel_rect.top + 20
        for i, mission in enumerate(visible_missions):
            mission_y = mission_start_y + i * 100
            
            # Mission box
            mission_rect = pygame.Rect(70, mission_y, mission_panel_rect.width - 40, 80)
            
            # Determine mission box color based on status and selection
            is_mission_hovered = (self.hover_button == f"mission_{mission.get('id', 'unknown')}")
            is_mission_selected = (self.selected_mission == mission.get('id'))
            is_unlocked = mission.get("unlocked", True) # Default to unlocked if key missing
            is_completed = mission.get("completed", False)

            if not is_unlocked:
                box_color = (60, 60, 60)  # Locked mission
                text_color = (150, 150, 150)
            elif is_completed:
                box_color = (40, 60, 40)  # Completed mission
                text_color = (180, 220, 180)
            elif is_mission_selected or is_mission_hovered:
                box_color = (70, 90, 70)  # Selected or hovered mission
                text_color = (220, 255, 220)
            else:
                box_color = (50, 70, 50)  # Available mission
                text_color = (200, 220, 200)
                
            pygame.draw.rect(screen, box_color, mission_rect, border_radius=5)
            
            # Add mission to clickable buttons if unlocked
            if is_unlocked:
                button_id = f"mission_{mission.get('id', 'unknown')}"
                self.buttons.append({"id": button_id, "rect": mission_rect})
                
            # Draw mission details (Name, Desc, Diff, Reward, Faction)
            name_pos = (mission_rect.left + 15, mission_rect.top + 10)
            # desc_pos = (mission_rect.left + 15, mission_rect.top + 35) # Description removed from list view
            diff_pos = (mission_rect.right - 150, mission_rect.top + 10) # Adjusted X
            reward_pos = (mission_rect.right - 150, mission_rect.top + 30) # Adjusted X and Y
            faction_pos = (mission_rect.right - 150, mission_rect.top + 50) # Adjusted X, New Y
            
            self.draw_text(screen, mission.get("name", "Unnamed Mission"), name_pos, "medium", text_color)
            # self.draw_text(screen, mission["description"][:60] + "...", desc_pos, "small", text_color)
            self.draw_text(screen, f"Diff: {mission.get('difficulty', 'N/A')}", diff_pos, "small", text_color)
            self.draw_text(screen, f"Reward: {mission.get('reward', 0)}", reward_pos, "small", text_color)
            # Draw Enemy Faction
            enemy_faction = mission.get("enemy_faction", "Unknown")
            self.draw_text(screen, f"Vs: {enemy_faction.capitalize()}", faction_pos, "small", text_color)
            
            # Status indicators (Completed/Locked)
            status_y = mission_rect.top + 35 # Position below name
            if is_completed:
                self.draw_text(screen, "[COMPLETED]", (mission_rect.left + 15, status_y), "small", (100, 200, 100))
            elif not is_unlocked:
                 self.draw_text(screen, "[LOCKED]", (mission_rect.left + 15, status_y), "small", (200, 100, 100))
        
        # Draw scroll indicators if needed
        if self.scroll_offset > 0:
            scroll_up_text = "▲"
            scroll_up_pos = (mission_panel_rect.centerx, mission_panel_rect.top + 15)
            self.draw_text(screen, scroll_up_text, scroll_up_pos, "medium", (150, 150, 150), True)
            
        if self.scroll_offset < max_scroll_filtered: # Use max_scroll based on filtered list
            scroll_down_text = "▼"
            scroll_down_pos = (mission_panel_rect.centerx, mission_panel_rect.bottom - 15)
            self.draw_text(screen, scroll_down_text, scroll_down_pos, "medium", (150, 150, 150), True)
        
        # Draw mission details if a mission is selected
        if self.selected_mission:
            selected_mission_data = next((m for m in self.mission_list if m.get("id") == self.selected_mission), None)
            if selected_mission_data:
                # Draw mission description panel
                desc_panel_rect = pygame.Rect(50, screen_height - 90, screen_width - 100, 70)
                pygame.draw.rect(screen, (50, 60, 50), desc_panel_rect, border_radius=5)
                
                # Draw mission description
                desc_pos = (desc_panel_rect.left + 20, desc_panel_rect.top + 15)
                self.draw_text(
                    screen, 
                    f"Location: {selected_mission_data.get('location', 'Unknown')}", 
                    desc_pos, 
                    "small", 
                    (200, 220, 200)
                )
                
                desc_pos = (desc_panel_rect.left + 20, desc_panel_rect.top + 40)
                self.draw_text(
                    screen, 
                    selected_mission_data.get("description", "No description available."), 
                    desc_pos, 
                    "small", 
                    (200, 220, 200)
                )
        
        # Draw control buttons
        button_width = 150
        button_height = 40
        
        # Start mission button
        if self.selected_mission:
            selected_mission_data_for_button = next((m for m in self.mission_list if m.get("id") == self.selected_mission), None)
            if selected_mission_data_for_button and selected_mission_data_for_button.get("unlocked", True):
                start_button_text = "START MISSION"
                if selected_mission_data_for_button["completed"]:
                    start_button_text = "REPLAY MISSION"
                
                start_button_rect = self.draw_button(
                    screen, start_button_text, 
                    (screen_width - 100, screen_height - 120),
                    (button_width, button_height),
                    hover=self.hover_button == "start_mission"
                )
                self.buttons.append({"id": "start_mission", "rect": start_button_rect})
        
        # Back button
        back_button_rect = self.draw_button(
            screen, "BACK", 
            (100, screen_height - 120),
            (button_width, button_height),
            hover=self.hover_button == "back"
        )
        self.buttons.append({"id": "back", "rect": back_button_rect})
        
        # Draw message if active
        if self.show_message:
            # Create semi-transparent overlay
            overlay = pygame.Surface((screen_width, screen_height), pygame.SRCALPHA)
            overlay.fill((0, 0, 0, 150))  # Black with 150 alpha
            screen.blit(overlay, (0, 0))
            
            # Draw message box
            message_box_width = 500
            message_box_height = 150
            message_box_rect = pygame.Rect(
                (screen_width - message_box_width) // 2,
                (screen_height - message_box_height) // 2,
                message_box_width,
                message_box_height
            )
            pygame.draw.rect(screen, (50, 50, 60), message_box_rect, border_radius=10)
            pygame.draw.rect(screen, (100, 100, 120), message_box_rect, border_radius=10, width=3)
            
            # Draw message text
            self.draw_text(
                screen,
                self.message_text,
                (message_box_rect.centerx, message_box_rect.centery - 20),
                "large",
                self.message_color,
                True
            )
            
            # Draw continue instruction
            self.draw_text(
                screen,
                "Press any key to continue",
                (message_box_rect.centerx, message_box_rect.centery + 30),
                "small",
                (180, 180, 180),
                True
            )
        
    def _handle_button_click(self, button_id):
        """Handle button clicks"""
        if button_id.startswith("filter_"): # Handle faction filter clicks
            new_filter = button_id.replace("filter_", "")
            if new_filter != self.selected_faction_filter:
                 self.selected_faction_filter = new_filter
                 self.scroll_offset = 0 # Reset scroll when filter changes
                 self.hover_button = None # Clear hover
                 self.last_action = f"Filtered missions by: {new_filter}"
                 print(f"Set mission filter to: {new_filter}")

        elif button_id.startswith("mission_"): # Handle mission selection clicks
            print(f"--- DEBUG: Handling mission click. Button ID: '{button_id}' ---")
            # Correctly extract the ID after the prefix
            prefix = "mission_"
            if button_id.startswith(prefix):
                 mission_id = button_id[len(prefix):]
            else:
                 # Should not happen if starts_with is true, but good practice
                 mission_id = None
                 print("--- ERROR: Button ID started with prefix but extraction failed?")

            print(f"--- DEBUG: Extracted Mission ID: '{mission_id}' ---")

            if mission_id:
                selected_mission_data = next((m for m in self.mission_list if m["id"] == mission_id), None)

                if selected_mission_data:
                     print(f"--- DEBUG: Found mission data for '{mission_id}'. Unlocked status: {selected_mission_data.get('unlocked')}")
                else:
                     print(f"--- DEBUG: Did NOT find mission data for '{mission_id}'.")

                # Check if unlocked
                if selected_mission_data and selected_mission_data.get("unlocked", True):
                    print(f"Selected mission: {selected_mission_data['name']} - PROCEEDING TO BATTLE")
                    self.selected_mission = mission_id # Store the ID for BattleScene
                    self.game_manager.change_state(GameState.BATTLE)
                else:
                    # Handle locked or invalid mission
                    print(f"Clicked locked or invalid mission: {mission_id} - NOT STARTING BATTLE")
                    self.message_text = "Mission is locked or unavailable."
                    self.message_color = (255, 100, 100)
                    self.show_message = True
                    self.message_timer = 0
            else:
                 print("--- ERROR: Failed to extract mission_id from button_id")

        elif button_id == "start_mission" and self.selected_mission:
             # This button probably doesn't exist anymore if clicking the mission directly starts it
             pass

        # Handle other potential buttons if added later (e.g., back button)

        elif button_id == "back":
            # Return to the main menu
            self.game_manager.change_state(GameState.MAIN_MENU) 