"""
Main Menu Scene Module
"""
import pygame
from game.ui.scene import Scene
from game.core.game_state import GameState

class MenuScene(Scene):
    """
    Main menu scene for the game
    Displays title and main options
    """
    def __init__(self, game_manager):
        """Initialize the menu scene"""
        super().__init__(game_manager)
        self.buttons = []
        self.hover_button = None
        self.show_settings = False
        self.show_achievements = False
        
        # Background color - dark blue with a hint of purple (Warhammer 40K theme)
        self.bg_color = (20, 20, 35)
        
        # Title with gradient effect parameters
        self.title_colors = [
            (180, 20, 40),  # Dark red
            (200, 40, 60),  # Medium red
            (220, 60, 80)   # Bright red
        ]
        self.title_cycle_speed = 0.05
        self.color_index = 0
        
        # Initialize menu buttons
        self._initialize_buttons()
        
    def enter(self):
        """Called when entering the menu scene"""
        # Reset button hover state
        self.hover_button = None
        
    def handle_event(self, event):
        """Handle pygame events"""
        if event.type == pygame.MOUSEMOTION:
            # Check for button hover
            mouse_pos = pygame.mouse.get_pos()
            self.hover_button = None
            
            # Check settings buttons if settings are shown
            if self.show_settings:
                for button_info in self.settings_buttons:
                    if self.is_point_inside_rect(mouse_pos, button_info["rect"]):
                        self.hover_button = button_info["id"]
                        break
            # Check achievements back button if achievements are shown
            elif self.show_achievements:
                if self.is_point_inside_rect(mouse_pos, self.achievements_back_button["rect"]):
                    self.hover_button = self.achievements_back_button["id"]
            else:
                # Otherwise check main menu buttons
                for button_info in self.buttons:
                    if self.is_point_inside_rect(mouse_pos, button_info["rect"]):
                        self.hover_button = button_info["id"]
                        break
        elif event.type == pygame.MOUSEBUTTONDOWN:
            if event.button == 1:  # Left mouse button
                mouse_pos = pygame.mouse.get_pos()
                
                # Check settings buttons if settings are shown
                if self.show_settings:
                    for button_info in self.settings_buttons:
                        if self.is_point_inside_rect(mouse_pos, button_info["rect"]):
                            button_info["action"]()
                            break
                # Check achievements back button if achievements are shown
                elif self.show_achievements:
                    if self.is_point_inside_rect(mouse_pos, self.achievements_back_button["rect"]):
                        self.achievements_back_button["action"]()
                else:
                    # Otherwise check main menu buttons
                    for button_info in self.buttons:
                        if self.is_point_inside_rect(mouse_pos, button_info["rect"]):
                            button_info["action"]()
                            break
        elif event.type == pygame.KEYDOWN:
            if event.key == pygame.K_SPACE or event.key == pygame.K_RETURN:
                if self.hover_button:
                    self._handle_button_click(self.hover_button)
            elif event.key == pygame.K_ESCAPE:
                # Quit the game when pressing Escape from the main menu
                pygame.event.post(pygame.event.Event(pygame.QUIT))
                
    def update(self):
        """Update the menu scene"""
        # Update title color cycle
        self.color_index = (self.color_index + self.title_cycle_speed) % len(self.title_colors)
        
    def render(self, screen):
        """Render the menu scene"""
        # Fill the background
        screen.fill(self.bg_color)
        
        # Draw the game title with gradient effect
        title_text = "WARHAMMER 40K"
        subtitle_text = "HEROES OF THE IMPERIUM"
        
        # Calculate color based on interpolation between two colors
        color_idx_int = int(self.color_index)
        color_idx_frac = self.color_index - color_idx_int
        color1 = self.title_colors[color_idx_int]
        color2 = self.title_colors[(color_idx_int + 1) % len(self.title_colors)]
        
        title_color = tuple(
            int(color1[i] * (1 - color_idx_frac) + color2[i] * color_idx_frac)
            for i in range(3)
        )
        
        screen_width, screen_height = screen.get_size()
        
        # Draw title
        title_pos = (screen_width // 2, screen_height // 4)
        self.draw_text(screen, title_text, title_pos, "title", title_color, True)
        
        # Draw subtitle
        subtitle_pos = (screen_width // 2, screen_height // 4 + 80)
        self.draw_text(screen, subtitle_text, subtitle_pos, "large", title_color, True)
        
        # Draw player level if there's save data
        level_text = f"Commander Level: {self.game_manager.player_data['level']}"
        level_surface = self.draw_text(screen, level_text, (screen_width - 10, 10), "small", (200, 200, 150), True)
        
        # If settings are shown, render settings menu
        if self.show_settings:
            # Draw settings panel background
            pygame.draw.rect(screen, (40, 40, 50), self.settings_panel)
            pygame.draw.rect(screen, (60, 60, 70), self.settings_panel, 2)
            
            # Draw settings title
            settings_title_font = pygame.font.SysFont(None, 48)
            settings_title = settings_title_font.render("Game Settings", True, (200, 200, 200))
            settings_title_rect = settings_title.get_rect(centerx=self.settings_panel.centerx, y=self.settings_panel.y + 20)
            screen.blit(settings_title, settings_title_rect)
            
            # Draw settings buttons
            for button_info in self.settings_buttons:
                color = button_info["hover_color"] if self.hover_button == button_info["id"] else button_info["color"]
                pygame.draw.rect(screen, color, button_info["rect"])
                pygame.draw.rect(screen, (100, 100, 100), button_info["rect"], 1)  # Border
                
                # Draw button text
                font = pygame.font.SysFont(None, 28)
                text_surface = font.render(button_info["text"], True, button_info["text_color"])
                text_rect = text_surface.get_rect(center=button_info["rect"].center)
                screen.blit(text_surface, text_rect)
        
        # If achievements are shown, render achievements screen
        elif self.show_achievements:
            # Draw achievements panel background
            pygame.draw.rect(screen, (40, 40, 50), self.achievements_panel)
            pygame.draw.rect(screen, (60, 60, 70), self.achievements_panel, 2)
            
            # Draw achievements title
            achievements_title_font = pygame.font.SysFont(None, 48)
            achievements_title = achievements_title_font.render("Achievements", True, (200, 200, 200))
            achievements_title_rect = achievements_title.get_rect(centerx=self.achievements_panel.centerx, y=self.achievements_panel.y + 20)
            screen.blit(achievements_title, achievements_title_rect)
            
            # Draw achievements list
            font = pygame.font.SysFont(None, 28)
            small_font = pygame.font.SysFont(None, 22)
            
            y_pos = self.achievements_panel.y + 80
            spacing = 85
            
            for ach_id, ach_data in self.game_manager.achievements.items():
                # Achievement box background - green if unlocked, darker if locked
                box_rect = pygame.Rect(
                    self.achievements_panel.x + 30,
                    y_pos,
                    self.achievements_panel.width - 60,
                    70
                )
                
                if ach_data["unlocked"]:
                    box_color = (50, 100, 50)  # Green for unlocked
                    text_color = (220, 220, 220)
                else:
                    box_color = (70, 70, 70)  # Gray for locked
                    text_color = (180, 180, 180)
                
                pygame.draw.rect(screen, box_color, box_rect)
                pygame.draw.rect(screen, (100, 100, 100), box_rect, 1)  # Border
                
                # Achievement title
                title_surface = font.render(ach_data["description"], True, text_color)
                screen.blit(title_surface, (box_rect.x + 10, box_rect.y + 10))
                
                # Achievement reward
                reward_text = f"Reward: {ach_data['reward']}"
                reward_surface = small_font.render(reward_text, True, text_color)
                screen.blit(reward_surface, (box_rect.x + 20, box_rect.y + 40))
                
                # Status text
                status_text = "UNLOCKED" if ach_data["unlocked"] else "LOCKED"
                status_surface = small_font.render(status_text, True, text_color)
                status_rect = status_surface.get_rect(right=box_rect.right - 10, y=box_rect.y + 40)
                screen.blit(status_surface, status_rect)
                
                y_pos += spacing
            
            # Draw back button
            button_info = self.achievements_back_button
            color = button_info["hover_color"] if self.hover_button == button_info["id"] else button_info["color"]
            pygame.draw.rect(screen, color, button_info["rect"])
            pygame.draw.rect(screen, (100, 100, 100), button_info["rect"], 1)  # Border
            
            # Draw button text
            text_surface = font.render(button_info["text"], True, button_info["text_color"])
            text_rect = text_surface.get_rect(center=button_info["rect"].center)
            screen.blit(text_surface, text_rect)
            
        else:
            # Draw main menu buttons
            for button_info in self.buttons:
                color = button_info["hover_color"] if self.hover_button == button_info["id"] else button_info["color"]
                pygame.draw.rect(screen, color, button_info["rect"])
                pygame.draw.rect(screen, (100, 100, 100), button_info["rect"], 1)  # Border
                
                # Draw button text
                font = pygame.font.SysFont(None, 32)
                text_surface = font.render(button_info["text"], True, button_info["text_color"])
                text_rect = text_surface.get_rect(center=button_info["rect"].center)
                screen.blit(text_surface, text_rect)
        
        # Draw footer text
        footer_text = "© 2023 Fan Project - Not affiliated with Games Workshop"
        footer_pos = (screen_width // 2, screen_height - 30)
        self.draw_text(screen, footer_text, footer_pos, "small", (150, 150, 150), True)
        
    def _handle_button_click(self, button_id):
        """Handle button clicks"""
        if button_id == "new_game":
            # Transition to mission select
            self.game_manager.change_state(GameState.MISSION_SELECT)
        elif button_id == "load_game":
            # TODO: Implement load game functionality
            print("Load game not implemented yet")
        elif button_id == "options":
            # TODO: Implement options menu
            print("Options not implemented yet")
        elif button_id == "quit":
            # Quit the game
            pygame.event.post(pygame.event.Event(pygame.QUIT)) 

    def _initialize_buttons(self):
        """Initialize menu buttons"""
        screen_width = self.game_manager.screen.get_width()
        screen_height = self.game_manager.screen.get_height()
        
        button_width = 250
        button_height = 60
        button_y_start = screen_height // 2
        button_spacing = 20
        
        # Main menu buttons with improved appearance
        self.buttons = [
            {
                "id": "start_game",
                "text": "Start Campaign",
                "rect": pygame.Rect(
                    (screen_width - button_width) // 2,
                    button_y_start,
                    button_width,
                    button_height
                ),
                "action": self._start_game,
                "color": (60, 60, 100),
                "hover_color": (80, 80, 120),
                "text_color": (220, 220, 220)
            },
            {
                "id": "settings",
                "text": "Settings",
                "rect": pygame.Rect(
                    (screen_width - button_width) // 2,
                    button_y_start + button_height + button_spacing,
                    button_width,
                    button_height
                ),
                "action": self._open_settings,
                "color": (60, 60, 100),
                "hover_color": (80, 80, 120),
                "text_color": (220, 220, 220)
            },
            {
                "id": "achievements",
                "text": "Achievements",
                "rect": pygame.Rect(
                    (screen_width - button_width) // 2,
                    button_y_start + (button_height + button_spacing) * 2,
                    button_width,
                    button_height
                ),
                "action": self._open_achievements,
                "color": (60, 60, 100),
                "hover_color": (80, 80, 120),
                "text_color": (220, 220, 220)
            },
            {
                "id": "game_detector",
                "text": "Game Detector",
                "rect": pygame.Rect(
                    (screen_width - button_width) // 2,
                    button_y_start + (button_height + button_spacing) * 3,
                    button_width,
                    button_height
                ),
                "action": self._open_game_detector,
                "color": (60, 60, 100),
                "hover_color": (80, 80, 120),
                "text_color": (220, 220, 220)
            },
            {
                "id": "quit",
                "text": "Quit Game",
                "rect": pygame.Rect(
                    (screen_width - button_width) // 2,
                    button_y_start + (button_height + button_spacing) * 4,
                    button_width,
                    button_height
                ),
                "action": self._quit_game,
                "color": (100, 60, 60),
                "hover_color": (120, 80, 80),
                "text_color": (220, 220, 220)
            }
        ]
        
        # Initialize settings screen
        self._initialize_settings_buttons()
        
        # Initialize achievements screen
        self._initialize_achievements_screen()

    def _start_game(self):
        """Start a new game"""
        self.game_manager.change_state(GameState.MISSION_SELECT)
        
    def _open_settings(self):
        """Open the settings menu"""
        self.show_settings = True
        self._initialize_settings_buttons()
        
    def _initialize_settings_buttons(self):
        """Initialize settings menu buttons"""
        button_width = 250
        button_height = 40
        button_spacing = 15
        
        # Settings panel background
        panel_width = 400
        panel_height = 500
        panel_x = (self.game_manager.screen.get_width() - panel_width) // 2
        panel_y = (self.game_manager.screen.get_height() - panel_height) // 2
        self.settings_panel = pygame.Rect(panel_x, panel_y, panel_width, panel_height)
        
        # Start positions for buttons within the panel
        start_x = panel_x + (panel_width - button_width) // 2
        start_y = panel_y + 80
        
        # Current settings
        settings = self.game_manager.settings
        
        # Settings buttons
        self.settings_buttons = [
            # Difficulty selector
            {
                "id": "difficulty",
                "text": f"Difficulty: {settings['difficulty']}",
                "rect": pygame.Rect(start_x, start_y, button_width, button_height),
                "color": (60, 60, 100),
                "hover_color": (80, 80, 120),
                "text_color": (255, 255, 255),
                "action": self._toggle_difficulty
            },
            # Sound volume
            {
                "id": "sound_volume",
                "text": f"Sound Volume: {int(settings['sound_volume'] * 100)}%",
                "rect": pygame.Rect(start_x, start_y + (button_height + button_spacing), button_width, button_height),
                "color": (60, 60, 100),
                "hover_color": (80, 80, 120),
                "text_color": (255, 255, 255),
                "action": self._adjust_sound_volume
            },
            # Music volume
            {
                "id": "music_volume",
                "text": f"Music Volume: {int(settings['music_volume'] * 100)}%",
                "rect": pygame.Rect(start_x, start_y + (button_height + button_spacing) * 2, button_width, button_height),
                "color": (60, 60, 100),
                "hover_color": (80, 80, 120),
                "text_color": (255, 255, 255),
                "action": self._adjust_music_volume
            },
            # Fullscreen toggle
            {
                "id": "fullscreen",
                "text": f"Fullscreen: {'ON' if settings['fullscreen'] else 'OFF'}",
                "rect": pygame.Rect(start_x, start_y + (button_height + button_spacing) * 3, button_width, button_height),
                "color": (60, 60, 100),
                "hover_color": (80, 80, 120),
                "text_color": (255, 255, 255),
                "action": self._toggle_fullscreen
            },
            # Show tooltips toggle
            {
                "id": "tooltips",
                "text": f"Show Tooltips: {'ON' if settings['show_tooltips'] else 'OFF'}",
                "rect": pygame.Rect(start_x, start_y + (button_height + button_spacing) * 4, button_width, button_height),
                "color": (60, 60, 100),
                "hover_color": (80, 80, 120),
                "text_color": (255, 255, 255),
                "action": self._toggle_tooltips
            },
            # Animations toggle
            {
                "id": "animations",
                "text": f"Enable Animations: {'ON' if settings['enable_animations'] else 'OFF'}",
                "rect": pygame.Rect(start_x, start_y + (button_height + button_spacing) * 5, button_width, button_height),
                "color": (60, 60, 100),
                "hover_color": (80, 80, 120),
                "text_color": (255, 255, 255),
                "action": self._toggle_animations
            },
            # Auto-save toggle
            {
                "id": "auto_save",
                "text": f"Auto-Save: {'ON' if settings['auto_save'] else 'OFF'}",
                "rect": pygame.Rect(start_x, start_y + (button_height + button_spacing) * 6, button_width, button_height),
                "color": (60, 60, 100),
                "hover_color": (80, 80, 120),
                "text_color": (255, 255, 255),
                "action": self._toggle_auto_save
            },
            # Back button
            {
                "id": "back_to_menu",
                "text": "Save & Return",
                "rect": pygame.Rect(start_x, start_y + (button_height + button_spacing) * 7, button_width, button_height),
                "color": (100, 60, 60),
                "hover_color": (120, 80, 80),
                "text_color": (255, 255, 255),
                "action": self._close_settings
            }
        ]
    
    def _toggle_difficulty(self):
        """Toggle between difficulty levels"""
        difficulties = ["Easy", "Normal", "Hard", "Impossible"]
        current_idx = difficulties.index(self.game_manager.settings["difficulty"])
        next_idx = (current_idx + 1) % len(difficulties)
        
        self.game_manager.change_settings("difficulty", difficulties[next_idx])
        self._initialize_settings_buttons()  # Refresh buttons to show new setting
    
    def _adjust_sound_volume(self):
        """Adjust sound volume"""
        current_volume = self.game_manager.settings["sound_volume"]
        # Increase by 0.1, loop back to 0 after 1.0
        new_volume = round((current_volume + 0.1) % 1.1, 1)
        if new_volume > 1.0:
            new_volume = 0.0
            
        self.game_manager.change_settings("sound_volume", new_volume)
        self._initialize_settings_buttons()
    
    def _adjust_music_volume(self):
        """Adjust music volume"""
        current_volume = self.game_manager.settings["music_volume"]
        # Increase by 0.1, loop back to 0 after 1.0
        new_volume = round((current_volume + 0.1) % 1.1, 1)
        if new_volume > 1.0:
            new_volume = 0.0
            
        self.game_manager.change_settings("music_volume", new_volume)
        self._initialize_settings_buttons()
    
    def _toggle_fullscreen(self):
        """Toggle fullscreen mode"""
        current = self.game_manager.settings["fullscreen"]
        self.game_manager.change_settings("fullscreen", not current)
        self._initialize_settings_buttons()
    
    def _toggle_tooltips(self):
        """Toggle showing tooltips"""
        current = self.game_manager.settings["show_tooltips"]
        self.game_manager.change_settings("show_tooltips", not current)
        self._initialize_settings_buttons()
    
    def _toggle_animations(self):
        """Toggle animations"""
        current = self.game_manager.settings["enable_animations"]
        self.game_manager.change_settings("enable_animations", not current)
        self._initialize_settings_buttons()
    
    def _toggle_auto_save(self):
        """Toggle auto-save"""
        current = self.game_manager.settings["auto_save"]
        self.game_manager.change_settings("auto_save", not current)
        self._initialize_settings_buttons()
    
    def _close_settings(self):
        """Close the settings menu and save settings"""
        self.show_settings = False
        self.game_manager.save_game()  # Save settings
        
    def _quit_game(self):
        """Quit the game"""
        self.game_manager.save_game()  # Save before quitting
        pygame.event.post(pygame.event.Event(pygame.QUIT)) 

    def _open_achievements(self):
        """Open the achievements screen"""
        self.show_achievements = True
        self._initialize_achievements_screen()
        
    def _initialize_achievements_screen(self):
        """Initialize achievements screen elements"""
        # Achievements panel background
        panel_width = 500
        panel_height = 500
        panel_x = (self.game_manager.screen.get_width() - panel_width) // 2
        panel_y = (self.game_manager.screen.get_height() - panel_height) // 2
        self.achievements_panel = pygame.Rect(panel_x, panel_y, panel_width, panel_height)
        
        # Back button dimensions and position
        button_width = 200
        button_height = 40
        button_x = panel_x + (panel_width - button_width) // 2
        button_y = panel_y + panel_height - button_height - 20
        
        # Create back button
        self.achievements_back_button = {
            "id": "back_from_achievements",
            "text": "Back to Menu",
            "rect": pygame.Rect(button_x, button_y, button_width, button_height),
            "color": (100, 60, 60),
            "hover_color": (120, 80, 80),
            "text_color": (255, 255, 255),
            "action": self._close_achievements
        } 

    def _close_achievements(self):
        """Close the achievements screen"""
        self.show_achievements = False

    def _open_game_detector(self):
        """Open the game detector terms screen"""
        # Show terms scene to accept/manage game detector
        self.game_manager.show_terms_scene() 
        
    def on_resize(self, screen):
        """Handle screen resize"""
        # Reinitialize buttons with new screen dimensions
        self._initialize_buttons()
        
        # If settings panel is open, reinitialize it
        if self.show_settings:
            self._initialize_settings_buttons()
            
        # If achievements panel is open, reinitialize it
        if self.show_achievements:
            self._initialize_achievements_screen() 