"""
Customization Scene Module
Handles unit and protagonist customization
"""
import pygame
from game.ui.scene import Scene
from game.core.game_state import GameState

class CustomizationScene(Scene):
    """
    Customization scene for upgrading the protagonist and units
    """
    def __init__(self, game_manager):
        """Initialize the customization scene"""
        super().__init__(game_manager)
        self.buttons = []
        self.hover_button = None
        self.selected_unit = None
        self.current_tab = "marine"  # "marine", "units", "inventory"
        
        # Background color - dark blue (Warhammer 40K imperial theme)
        self.bg_color = (20, 20, 35)
        
    def enter(self):
        """Called when entering the customization scene"""
        # Reset state
        self.hover_button = None
        self.selected_unit = None
        
    def handle_event(self, event):
        """Handle pygame events"""
        if event.type == pygame.MOUSEMOTION:
            # Check for button hover
            mouse_pos = pygame.mouse.get_pos()
            self.hover_button = None
            for button_info in self.buttons:
                if self.is_point_inside_rect(mouse_pos, button_info["rect"]):
                    self.hover_button = button_info["id"]
                    break
        
        elif event.type == pygame.MOUSEBUTTONDOWN:
            if event.button == 1:  # Left mouse button
                mouse_pos = pygame.mouse.get_pos()
                for button_info in self.buttons:
                    if self.is_point_inside_rect(mouse_pos, button_info["rect"]):
                        self._handle_button_click(button_info["id"])
                        break
        
        elif event.type == pygame.KEYDOWN:
            if event.key == pygame.K_ESCAPE:
                # Return to the main menu
                self.game_manager.change_state(GameState.MAIN_MENU)
                
    def update(self):
        """Update the customization scene"""
        pass
        
    def render(self, screen):
        """Render the customization scene"""
        # Fill the background
        screen.fill(self.bg_color)
        
        screen_width, screen_height = screen.get_size()
        
        # Draw the header
        header_text = "CUSTOMIZATION"
        header_pos = (screen_width // 2, 50)
        self.draw_text(screen, header_text, header_pos, "large", (200, 200, 200), True)
        
        # Reset the buttons list
        self.buttons = []
        
        # Draw tabs
        self._draw_tabs(screen)
        
        # Draw content based on current tab
        if self.current_tab == "marine":
            self._draw_marine_tab(screen)
        elif self.current_tab == "units":
            self._draw_units_tab(screen)
        elif self.current_tab == "inventory":
            self._draw_inventory_tab(screen)
        
        # Draw back button
        back_button_rect = self.draw_button(
            screen, "BACK", 
            (100, screen_height - 50),
            (150, 40),
            hover=self.hover_button == "back"
        )
        self.buttons.append({"id": "back", "rect": back_button_rect})
        
    def _draw_tabs(self, screen):
        """Draw the navigation tabs"""
        screen_width = screen.get_size()[0]
        
        tab_width = 200
        tab_height = 40
        tab_spacing = 20
        tabs_total_width = 3 * tab_width + 2 * tab_spacing
        
        tab_y = 100
        tab_start_x = (screen_width - tabs_total_width) // 2
        
        # Space Marine tab
        marine_tab_rect = pygame.Rect(tab_start_x, tab_y, tab_width, tab_height)
        marine_tab_active = self.current_tab == "marine"
        marine_tab_hover = self.hover_button == "tab_marine"
        
        pygame.draw.rect(
            screen, 
            (60, 60, 100) if marine_tab_active else (40, 40, 60),
            marine_tab_rect,
            border_radius=5
        )
        
        # Tab border
        pygame.draw.rect(
            screen, 
            (100, 100, 150) if marine_tab_active or marine_tab_hover else (80, 80, 100),
            marine_tab_rect,
            border_radius=5,
            width=2
        )
        
        # Tab text
        self.draw_text(
            screen,
            "SPACE MARINE",
            marine_tab_rect.center,
            "medium",
            (200, 200, 255) if marine_tab_active else (150, 150, 200),
            True
        )
        
        self.buttons.append({"id": "tab_marine", "rect": marine_tab_rect})
        
        # Units tab
        units_tab_rect = pygame.Rect(tab_start_x + tab_width + tab_spacing, tab_y, tab_width, tab_height)
        units_tab_active = self.current_tab == "units"
        units_tab_hover = self.hover_button == "tab_units"
        
        pygame.draw.rect(
            screen, 
            (60, 60, 100) if units_tab_active else (40, 40, 60),
            units_tab_rect,
            border_radius=5
        )
        
        # Tab border
        pygame.draw.rect(
            screen, 
            (100, 100, 150) if units_tab_active or units_tab_hover else (80, 80, 100),
            units_tab_rect,
            border_radius=5,
            width=2
        )
        
        # Tab text
        self.draw_text(
            screen,
            "UNITS",
            units_tab_rect.center,
            "medium",
            (200, 200, 255) if units_tab_active else (150, 150, 200),
            True
        )
        
        self.buttons.append({"id": "tab_units", "rect": units_tab_rect})
        
        # Inventory tab
        inventory_tab_rect = pygame.Rect(tab_start_x + 2 * (tab_width + tab_spacing), tab_y, tab_width, tab_height)
        inventory_tab_active = self.current_tab == "inventory"
        inventory_tab_hover = self.hover_button == "tab_inventory"
        
        pygame.draw.rect(
            screen, 
            (60, 60, 100) if inventory_tab_active else (40, 40, 60),
            inventory_tab_rect,
            border_radius=5
        )
        
        # Tab border
        pygame.draw.rect(
            screen, 
            (100, 100, 150) if inventory_tab_active or inventory_tab_hover else (80, 80, 100),
            inventory_tab_rect,
            border_radius=5,
            width=2
        )
        
        # Tab text
        self.draw_text(
            screen,
            "INVENTORY",
            inventory_tab_rect.center,
            "medium",
            (200, 200, 255) if inventory_tab_active else (150, 150, 200),
            True
        )
        
        self.buttons.append({"id": "tab_inventory", "rect": inventory_tab_rect})
    
    def _draw_marine_tab(self, screen):
        """Draw the Space Marine customization tab"""
        screen_width, screen_height = screen.get_size()
        
        # Main content area
        content_rect = pygame.Rect(50, 160, screen_width - 100, screen_height - 220)
        pygame.draw.rect(screen, (30, 30, 50), content_rect, border_radius=5)
        
        # Marine preview area (left side)
        preview_rect = pygame.Rect(content_rect.left + 20, content_rect.top + 20, 300, 400)
        pygame.draw.rect(screen, (40, 40, 60), preview_rect, border_radius=5)
        
        # Draw placeholder space marine silhouette
        marine_rect = pygame.Rect(0, 0, 200, 300)
        marine_rect.center = preview_rect.center
        pygame.draw.rect(screen, (0, 0, 150), marine_rect, border_radius=10)
        
        # Draw marine stats and equipment (right side)
        stats_rect = pygame.Rect(preview_rect.right + 40, preview_rect.top, 
                                content_rect.right - preview_rect.right - 60, preview_rect.height)
        
        # Header
        self.draw_text(
            screen,
            "SPACE MARINE COMMANDER",
            (stats_rect.left + stats_rect.width // 2, stats_rect.top + 20),
            "medium",
            (200, 200, 220),
            True
        )
        
        # Stats
        stats_box = pygame.Rect(stats_rect.left, stats_rect.top + 60, stats_rect.width, 120)
        pygame.draw.rect(screen, (40, 40, 70), stats_box, border_radius=5)
        
        # Stats text
        self.draw_text(screen, "Level: 1", (stats_box.left + 20, stats_box.top + 20), "medium", (200, 200, 220))
        self.draw_text(screen, "Experience: 0/100", (stats_box.left + 20, stats_box.top + 50), "medium", (200, 200, 220))
        self.draw_text(screen, "Health: 150", (stats_box.left + 20, stats_box.top + 80), "medium", (200, 200, 220))
        
        # Equipment slots
        equip_y = stats_box.bottom + 30
        
        # Weapon slot
        weapon_box = pygame.Rect(stats_rect.left, equip_y, stats_rect.width, 60)
        pygame.draw.rect(screen, (40, 40, 70), weapon_box, border_radius=5)
        self.draw_text(screen, "Weapon: Standard Bolter", (weapon_box.left + 20, weapon_box.top + 20), "medium", (200, 200, 220))
        
        # Armor slot
        armor_box = pygame.Rect(stats_rect.left, equip_y + 70, stats_rect.width, 60)
        pygame.draw.rect(screen, (40, 40, 70), armor_box, border_radius=5)
        self.draw_text(screen, "Armor: Standard Power Armor", (armor_box.left + 20, armor_box.top + 20), "medium", (200, 200, 220))
        
        # Artifact slot
        artifact_box = pygame.Rect(stats_rect.left, equip_y + 140, stats_rect.width, 60)
        pygame.draw.rect(screen, (40, 40, 70), artifact_box, border_radius=5)
        self.draw_text(screen, "Artifact: None", (artifact_box.left + 20, artifact_box.top + 20), "medium", (200, 200, 220))
        
        # Customize buttons
        button_width = 160
        button_height = 40
        button_spacing = 20
        buttons_y = content_rect.bottom - 60
        
        # Customize weapon button
        weapon_button_rect = self.draw_button(
            screen, "CHANGE WEAPON", 
            (content_rect.left + button_width // 2 + 40, buttons_y),
            (button_width, button_height),
            hover=self.hover_button == "customize_weapon",
            centered=True
        )
        self.buttons.append({"id": "customize_weapon", "rect": weapon_button_rect})
        
        # Customize armor button
        armor_button_rect = self.draw_button(
            screen, "CHANGE ARMOR", 
            (content_rect.left + button_width + button_spacing + button_width // 2 + 40, buttons_y),
            (button_width, button_height),
            hover=self.hover_button == "customize_armor",
            centered=True
        )
        self.buttons.append({"id": "customize_armor", "rect": armor_button_rect})
        
        # Customize artifact button
        artifact_button_rect = self.draw_button(
            screen, "CHANGE ARTIFACT", 
            (content_rect.left + 2 * (button_width + button_spacing) + button_width // 2 + 40, buttons_y),
            (button_width, button_height),
            hover=self.hover_button == "customize_artifact",
            centered=True
        )
        self.buttons.append({"id": "customize_artifact", "rect": artifact_button_rect})
    
    def _draw_units_tab(self, screen):
        """Draw the Units customization tab"""
        screen_width, screen_height = screen.get_size()
        
        # Main content area
        content_rect = pygame.Rect(50, 160, screen_width - 100, screen_height - 220)
        pygame.draw.rect(screen, (30, 30, 50), content_rect, border_radius=5)
        
        # Title
        self.draw_text(
            screen,
            "YOUR UNITS",
            (content_rect.left + content_rect.width // 2, content_rect.top + 30),
            "medium",
            (200, 200, 220),
            True
        )
        
        # Message for prototype
        self.draw_text(
            screen,
            "Unit customization not available in prototype",
            (content_rect.left + content_rect.width // 2, content_rect.top + content_rect.height // 2),
            "medium",
            (150, 150, 180),
            True
        )
    
    def _draw_inventory_tab(self, screen):
        """Draw the Inventory tab"""
        screen_width, screen_height = screen.get_size()
        
        # Main content area
        content_rect = pygame.Rect(50, 160, screen_width - 100, screen_height - 220)
        pygame.draw.rect(screen, (30, 30, 50), content_rect, border_radius=5)
        
        # Title
        self.draw_text(
            screen,
            "INVENTORY",
            (content_rect.left + content_rect.width // 2, content_rect.top + 30),
            "medium",
            (200, 200, 220),
            True
        )
        
        # Message for prototype
        self.draw_text(
            screen,
            "Inventory system not available in prototype",
            (content_rect.left + content_rect.width // 2, content_rect.top + content_rect.height // 2),
            "medium",
            (150, 150, 180),
            True
        )
    
    def _handle_button_click(self, button_id):
        """Handle button clicks"""
        if button_id.startswith("tab_"):
            # Switch tabs
            tab_name = button_id[4:]  # Remove "tab_" prefix
            self.current_tab = tab_name
            
        elif button_id == "back":
            # Return to the main menu
            self.game_manager.change_state(GameState.MAIN_MENU)
            
        elif button_id == "customize_weapon":
            # Weapon customization (prototype message)
            print("Weapon customization not implemented in prototype")
            
        elif button_id == "customize_armor":
            # Armor customization (prototype message)
            print("Armor customization not implemented in prototype")
            
        elif button_id == "customize_artifact":
            # Artifact customization (prototype message)
            print("Artifact customization not implemented in prototype") 