"""
Unit Loader Module
Loads unit templates from data files and creates unit instances.
"""
import os
import json
import copy
import random

class UnitLoader:
    """
    Loads unit templates from JSON files and creates instances.
    """
    def __init__(self, data_path="game/data/units"):
        """Initialize the loader and load unit templates."""
        self.data_path = data_path
        self.unit_templates = {}
        self.load_templates()

    def load_templates(self):
        """Load all unit templates from JSON files in the data path."""
        print("Loading unit templates...")
        self.unit_templates = {}
        if not os.path.exists(self.data_path):
            print(f"Warning: Unit data path not found: {self.data_path}")
            return

        for filename in os.listdir(self.data_path):
            if filename.endswith(".json"):
                faction_name = filename.replace(".json", "")
                file_path = os.path.join(self.data_path, filename)
                try:
                    with open(file_path, 'r') as f:
                        data = json.load(f)
                        if faction_name not in self.unit_templates:
                            self.unit_templates[faction_name] = {}
                        # Store each unit type under the faction
                        for unit_type, template in data.items():
                            if 'faction' not in template:
                                template['faction'] = faction_name # Ensure faction is set
                            self.unit_templates[faction_name][unit_type] = template
                            print(f"  Loaded template: {faction_name} - {unit_type}")
                except json.JSONDecodeError as e:
                    print(f"Error decoding JSON from {filename}: {e}")
                except Exception as e:
                    print(f"Error loading unit template file {filename}: {e}")
        print("Unit template loading complete.")

    def get_template(self, unit_type, faction):
        """Get the template for a specific unit type and faction."""
        return self.unit_templates.get(faction, {}).get(unit_type)

    def create_unit_instance(self, unit_type, faction, instance_id, position):
        """
        Create a new unit instance dictionary from a template.

        Args:
            unit_type (str): The type ID of the unit (e.g., 'tactical', 'boy').
            faction (str): The faction of the unit (e.g., 'space_marines').
            instance_id (str): A unique ID for this specific instance (e.g., 'player_1', 'enemy_ork_boy_3').
            position (tuple): The starting (x, y) grid position.

        Returns:
            dict: A new dictionary representing the unit instance, or None if template not found.
        """
        template = self.get_template(unit_type, faction)
        if not template:
            print(f"Warning: Unit template not found for {faction} - {unit_type}")
            # Optionally create a default placeholder unit here?
            return None

        # Create a deep copy to avoid modifying the template
        instance = copy.deepcopy(template)

        # Set instance-specific properties
        instance["id"] = instance_id
        instance["position"] = position
        instance["hp"] = instance.get("max_hp", 1) # Start at full health
        instance["unit_type"] = unit_type # Ensure unit_type is stored
        instance["used_abilities"] = set() # Initialize empty set for used abilities
        
        # Convert size list back to tuple if needed (JSON saves as list)
        if "size" in instance and isinstance(instance["size"], list):
            instance["size"] = tuple(instance["size"])
        else:
            instance.setdefault("size", (1, 1)) # Ensure size exists, default to 1x1

        # Add other default fields if they might be missing from JSON
        instance.setdefault("attack_damage_type", "normal")
        instance.setdefault("is_weak_to_explosives", False)
        instance.setdefault("color", (random.randint(50, 200), random.randint(50, 200), random.randint(50, 200))) # Random color if not in template

        return instance 