"""
Unit Factory Module
Handles the creation of units using templates
"""
from game.core.unit import Unit
from game.data.units_data import UNIT_TEMPLATES, FACTIONS, UNIT_TYPES

class UnitFactory:
    """
    Factory class for creating units based on templates
    """
    @staticmethod
    def create_unit_from_template(template_id, name=None, level=None):
        """
        Create a unit from a template ID
        
        Args:
            template_id: ID of the template in UNIT_TEMPLATES
            name: Optional custom name (if None, uses template name)
            level: Optional level (if None, uses template base_level)
            
        Returns:
            Unit: A new unit instance based on the template
        """
        if template_id not in UNIT_TEMPLATES:
            raise ValueError(f"Unit template '{template_id}' not found")
            
        template = UNIT_TEMPLATES[template_id]
        
        # Use provided values or defaults from template
        actual_name = name or template["name"]
        actual_level = level or template["base_level"]
        
        # Create the unit
        unit = Unit(
            actual_name,
            template["faction"],
            template["unit_type"],
            level=actual_level
        )
        
        # Add custom abilities if specified in template
        if "abilities" in template:
            unit.abilities = template["abilities"]
            
        return unit
    
    @staticmethod
    def create_custom_unit(name, faction, unit_type, level=1, custom_abilities=None):
        """
        Create a custom unit with the specified properties
        
        Args:
            name: Unit name
            faction: Faction ID (must exist in FACTIONS)
            unit_type: Unit type ID (must exist in UNIT_TYPES)
            level: Unit level (default: 1)
            custom_abilities: Optional list of custom abilities
            
        Returns:
            Unit: A new custom unit instance
        """
        if faction not in FACTIONS:
            raise ValueError(f"Faction '{faction}' not found")
            
        if unit_type not in UNIT_TYPES:
            raise ValueError(f"Unit type '{unit_type}' not found")
            
        # Create the unit
        unit = Unit(name, faction, unit_type, level=level)
        
        # Override abilities if provided
        if custom_abilities:
            unit.abilities = custom_abilities
            
        return unit 