"""
Unit Module
Defines the unit class and related functionality
"""
import uuid

class Unit:
    """
    Represents a unit in the game
    """
    def __init__(self, name, faction, unit_type, level=1):
        """Initialize a unit with basic attributes"""
        self.id = str(uuid.uuid4())
        self.name = name
        self.faction = faction
        self.unit_type = unit_type
        self.level = level
        
        # Set base stats based on faction and unit type
        self.base_stats = self._get_base_stats()
        
        # Actual stats (modified by level, equipment, etc.)
        self.stats = self.base_stats.copy()
        self._recalculate_stats()
        
        # Current state
        self.current_hp = self.stats["hp"]
        
        # Equipment and abilities
        self.equipment = {
            "weapon": None,
            "armor": None,
            "accessory": None
        }
        
        self.abilities = self._get_default_abilities()
        
    def _get_base_stats(self):
        """Get base stats based on faction and unit type"""
        # Default stats
        base_stats = {
            "hp": 100,
            "attack": 10,
            "defense": 10,
            "speed": 10,
            "move_range": 3,
            "attack_range": 1
        }
        
        # Modify based on faction
        if self.faction == "space_marines":
            base_stats["hp"] += 20
            base_stats["attack"] += 5
            base_stats["defense"] += 5
        elif self.faction == "chaos_marines":
            base_stats["hp"] += 10
            base_stats["attack"] += 10
            base_stats["speed"] -= 1
        elif self.faction == "eldar":
            base_stats["hp"] -= 10
            base_stats["attack"] += 5
            base_stats["speed"] += 3
            base_stats["move_range"] += 1
        elif self.faction == "orks":
            base_stats["hp"] += 30
            base_stats["attack"] += 5
            base_stats["defense"] -= 5
        elif self.faction == "tau":
            base_stats["hp"] -= 20
            base_stats["attack"] += 10
            base_stats["attack_range"] += 2
        elif self.faction == "necrons":
            base_stats["hp"] += 10
            base_stats["defense"] += 10
            base_stats["speed"] -= 2
            
        # Modify based on unit type
        if self.unit_type == "melee":
            base_stats["attack"] += 10
            base_stats["defense"] += 5
            base_stats["attack_range"] = 1
        elif self.unit_type == "ranged":
            base_stats["attack"] += 5
            base_stats["attack_range"] = 3
        elif self.unit_type == "heavy":
            base_stats["hp"] += 30
            base_stats["attack"] += 15
            base_stats["defense"] += 10
            base_stats["speed"] -= 1
            base_stats["move_range"] -= 1
        elif self.unit_type == "scout":
            base_stats["hp"] -= 20
            base_stats["speed"] += 2
            base_stats["move_range"] += 2
        elif self.unit_type == "elite":
            base_stats["hp"] += 20
            base_stats["attack"] += 10
            base_stats["defense"] += 10
            base_stats["speed"] += 1
            
        return base_stats
    
    def _get_default_abilities(self):
        """Get default abilities based on faction and unit type"""
        abilities = []
        
        # Faction abilities
        if self.faction == "space_marines":
            abilities.append("And They Shall Know No Fear")
        elif self.faction == "chaos_marines":
            abilities.append("Mark of Chaos")
        elif self.faction == "eldar":
            abilities.append("Ancient Reflexes")
        elif self.faction == "orks":
            abilities.append("Waaagh!")
        elif self.faction == "tau":
            abilities.append("For the Greater Good")
        elif self.faction == "necrons":
            abilities.append("Living Metal")
            
        # Unit type abilities
        if self.unit_type == "melee":
            abilities.append("Charge")
        elif self.unit_type == "ranged":
            abilities.append("Overwatch")
        elif self.unit_type == "heavy":
            abilities.append("Suppressive Fire")
        elif self.unit_type == "scout":
            abilities.append("Infiltrate")
        elif self.unit_type == "elite":
            abilities.append("Heroic Intervention")
            
        return abilities
    
    def _recalculate_stats(self):
        """Recalculate stats based on level and equipment"""
        # Start with base stats
        self.stats = self.base_stats.copy()
        
        # Apply level bonuses (5% increase per level)
        level_multiplier = 1 + 0.05 * (self.level - 1)
        for stat in self.stats:
            self.stats[stat] = int(self.stats[stat] * level_multiplier)
            
        # Apply equipment bonuses (to be implemented)
        # TODO: Implement equipment stat bonuses
    
    def level_up(self):
        """Level up the unit"""
        self.level += 1
        old_hp = self.stats["hp"]
        
        # Recalculate stats
        self._recalculate_stats()
        
        # Heal by the difference in max HP
        hp_increase = self.stats["hp"] - old_hp
        self.current_hp = min(self.current_hp + hp_increase, self.stats["hp"])
        
        return f"{self.name} reached level {self.level}!"
    
    def equip(self, item, slot):
        """Equip an item in the specified slot"""
        if slot in self.equipment:
            old_item = self.equipment[slot]
            self.equipment[slot] = item
            
            # Recalculate stats
            self._recalculate_stats()
            
            return old_item
        else:
            raise ValueError(f"Invalid equipment slot: {slot}")
    
    def take_damage(self, amount):
        """
        Take damage and return True if unit is defeated
        """
        self.current_hp = max(0, self.current_hp - amount)
        return self.current_hp <= 0
    
    def heal(self, amount):
        """Heal the unit"""
        self.current_hp = min(self.current_hp + amount, self.stats["hp"])
        
    def is_defeated(self):
        """Check if the unit is defeated"""
        return self.current_hp <= 0
    
    def to_dict(self):
        """Convert unit to dictionary for serialization"""
        return {
            "id": self.id,
            "name": self.name,
            "faction": self.faction,
            "unit_type": self.unit_type,
            "level": self.level,
            "stats": self.stats,
            "current_hp": self.current_hp,
            "equipment": self.equipment,
            "abilities": self.abilities
        }
    
    @classmethod
    def from_dict(cls, data):
        """Create a unit from dictionary data"""
        unit = cls(data["name"], data["faction"], data["unit_type"], data["level"])
        unit.id = data["id"]
        unit.stats = data["stats"]
        unit.current_hp = data["current_hp"]
        unit.equipment = data["equipment"]
        unit.abilities = data["abilities"]
        return unit
    
    def __str__(self):
        """String representation of the unit"""
        return f"{self.name} (Lvl {self.level} {self.faction} {self.unit_type})" 