"""
Sprite Module
Handles sprite rendering and animation
"""
import pygame
import os

class Sprite:
    """
    Represents a sprite for game entities with animation capabilities
    """
    def __init__(self, image=None, position=(0, 0), scale=1.0, target_size=None):
        """
        Initialize a sprite
        
        Args:
            image: Surface or path to image file
            position: (x, y) position on screen
            scale: Scale factor (used if target_size is None)
            target_size: Optional (width, height) tuple for fixed size
        """
        self.position = position
        self.scale = scale
        self.target_size = target_size
        self.angle = 0
        self.flip_x = False
        self.flip_y = False
        self.alpha = 255  # Full opacity
        self.visible = True
        
        # Animation properties
        self.frames = []
        self.current_frame = 0
        self.animation_speed = 0.1  # Frames per update
        self.animation_timer = 0
        self.is_animating = False
        self.animation_loop = True
        
        # Set image if provided
        if image:
            if isinstance(image, str):
                self.load_image(image)
            else:
                self.set_image(image)
        else:
            # Create a placeholder image
            self.image = self._create_placeholder()
            self.original_image = self.image.copy()
            self.rect = self.image.get_rect()
            self.rect.topleft = self.position
    
    def _create_placeholder(self, color=(128, 128, 128)):
        """Create a placeholder image using target_size or default size"""
        width, height = self.target_size if self.target_size else (64, 64)
        
        surface = pygame.Surface((width, height), pygame.SRCALPHA)
        surface.fill((0, 0, 0, 0))  # Transparent
        
        # Draw a colored rectangle
        inner_rect = pygame.Rect(2, 2, width - 4, height - 4)
        pygame.draw.rect(surface, color, inner_rect, border_radius=8)
        
        # Draw an X
        pygame.draw.line(surface, (200, 200, 200), (10, 10), (width - 10, height - 10), 2)
        pygame.draw.line(surface, (200, 200, 200), (width - 10, 10), (10, height - 10), 2)
        
        return surface
    
    def load_image(self, file_path):
        """
        Load an image from file and resize based on target_size or scale
        
        Args:
            file_path: Path to the image file
        """
        try:
            if os.path.exists(file_path):
                loaded_image = pygame.image.load(file_path).convert_alpha()
                self.original_image_unscaled = loaded_image.copy()
                
                # Apply target_size or scale
                if self.target_size:
                    self.image = pygame.transform.smoothscale(loaded_image, self.target_size)
                elif self.scale != 1.0:
                    new_width = int(loaded_image.get_width() * self.scale)
                    new_height = int(loaded_image.get_height() * self.scale)
                    self.image = pygame.transform.smoothscale(loaded_image, (new_width, new_height))
                else:
                    self.image = loaded_image
                
                self.original_image = self.image.copy()
                self.rect = self.image.get_rect()
                
                # Always use topleft positioning
                self.rect.topleft = self.position
            else:
                # Create a placeholder if file doesn't exist
                print(f"Image file not found: {file_path}")
                self.image = self._create_placeholder(color=(200, 50, 50))
                self.original_image = self.image.copy()
                self.rect = self.image.get_rect()
                self.rect.topleft = self.position
        except pygame.error as e:
            # Create a placeholder on error
            print(f"Error loading image '{file_path}': {e}")
            self.image = self._create_placeholder(color=(200, 50, 50))
            self.original_image = self.image.copy()
            self.rect = self.image.get_rect()
            self.rect.topleft = self.position
    
    def set_image(self, surface):
        """
        Set sprite image from a pygame Surface and resize based on target_size or scale
        
        Args:
            surface: Pygame Surface to use as the sprite image
        """
        if surface is None:
            # Create a placeholder instead
            self.image = self._create_placeholder()
            self.original_image = self.image.copy()
            self.rect = self.image.get_rect()
            # Ensure position is a tuple of integers
            if not isinstance(self.position, tuple) or len(self.position) != 2:
                self.position = (0, 0)
            self.rect.topleft = self.position
            return
            
        self.original_image_unscaled = surface.copy()
        
        # Apply target_size or scale
        if self.target_size:
            self.image = pygame.transform.smoothscale(surface, self.target_size)
        elif self.scale != 1.0:
            new_width = int(surface.get_width() * self.scale)
            new_height = int(surface.get_height() * self.scale)
            self.image = pygame.transform.smoothscale(surface, (new_width, new_height))
        else:
            self.image = surface
        
        self.original_image = self.image.copy()
        self.rect = self.image.get_rect()
        
        # Ensure position is a tuple of integers before setting topleft
        if not isinstance(self.position, tuple) or len(self.position) != 2:
            self.position = (0, 0)
        
        # Always use topleft positioning
        self.rect.topleft = self.position
    
    def set_position(self, x, y):
        """
        Set the sprite position
        
        Args:
            x: X coordinate (screen coordinates)
            y: Y coordinate (screen coordinates)
        """
        self.position = (x, y)
        if hasattr(self, 'rect'):
            # Use topleft positioning consistently for all sprites
            self.rect.topleft = self.position
    
    def set_scale(self, scale):
        """
        Set the sprite scale. Ignored if target_size is set.
        
        Args:
            scale: Scale factor for the image
        """
        # Only apply scale if target_size is not defined
        if self.target_size is None:
            self.scale = scale
            if hasattr(self, 'original_image_unscaled'):
                new_width = int(self.original_image_unscaled.get_width() * self.scale)
                new_height = int(self.original_image_unscaled.get_height() * self.scale)
                self.image = pygame.transform.smoothscale(self.original_image_unscaled, (new_width, new_height))
                self.original_image = self.image.copy()
                old_topleft = self.rect.topleft if hasattr(self, 'rect') else self.position
                self.rect = self.image.get_rect()
                self.rect.topleft = old_topleft
            else:
                # Fallback if original unscaled not available (e.g., only placeholder exists)
                if hasattr(self, 'original_image'):
                    new_width = int(self.original_image.get_width() * self.scale)
                    new_height = int(self.original_image.get_height() * self.scale)
                    self.image = pygame.transform.smoothscale(self.original_image, (new_width, new_height))
                    self.original_image = self.image.copy()
                    old_topleft = self.rect.topleft if hasattr(self, 'rect') else self.position
                    self.rect = self.image.get_rect()
                    self.rect.topleft = old_topleft
        else:
            print("Warning: set_scale ignored because target_size is set.")
    
    def set_rotation(self, angle):
        """
        Set the sprite rotation angle while maintaining the sprite's topleft position
        
        Args:
            angle: Rotation angle in degrees
        """
        # Only update if angle has changed
        if self.angle == angle:
            return
        
        self.angle = angle
        if hasattr(self, 'original_image'):
            # Save the original topleft position
            old_topleft = self.rect.topleft if hasattr(self, 'rect') else self.position
            
            # Rotate the image
            self.image = pygame.transform.rotate(self.original_image, self.angle)
            
            # Update rect with new dimensions
            self.rect = self.image.get_rect()
            
            # Restore the topleft position
            self.rect.topleft = old_topleft
    
    def set_flip(self, flip_x, flip_y):
        """
        Set whether the sprite should be flipped
        
        Args:
            flip_x: Flip horizontally
            flip_y: Flip vertically
        """
        self.flip_x = flip_x
        self.flip_y = flip_y
        if hasattr(self, 'original_image'):
            self.image = pygame.transform.flip(self.original_image, self.flip_x, self.flip_y)
            old_topleft = self.rect.topleft
            self.rect = self.image.get_rect()
            self.rect.topleft = old_topleft
    
    def set_alpha(self, alpha):
        """
        Set sprite transparency
        
        Args:
            alpha: Alpha value (0-255)
        """
        self.alpha = max(0, min(255, alpha))
        if hasattr(self, 'image'):
            self.image.set_alpha(self.alpha)
    
    def add_animation_frame(self, frame):
        """
        Add a frame to the sprite's animation, resizing if needed
        
        Args:
            frame: Surface to add as an animation frame
        """
        processed_frame = frame
        if self.target_size:
            processed_frame = pygame.transform.smoothscale(frame, self.target_size)
        elif self.scale != 1.0:
            new_width = int(frame.get_width() * self.scale)
            new_height = int(frame.get_height() * self.scale)
            processed_frame = pygame.transform.smoothscale(frame, (new_width, new_height))
        
        self.frames.append(processed_frame)
        # If this is the first frame and no image exists, set it
        if not self.image and len(self.frames) == 1:
            self.set_image(self.frames[0])
    
    def load_animation_frames(self, file_paths):
        """
        Load animation frames, resizing based on target_size or scale
        
        Args:
            file_paths: List of paths to frame images
        """
        self.frames = []
        for path in file_paths:
            try:
                if os.path.exists(path):
                    frame = pygame.image.load(path).convert_alpha()
                    processed_frame = frame
                    if self.target_size:
                        processed_frame = pygame.transform.smoothscale(frame, self.target_size)
                    elif self.scale != 1.0:
                        new_width = int(frame.get_width() * self.scale)
                        new_height = int(frame.get_height() * self.scale)
                        processed_frame = pygame.transform.smoothscale(frame, (new_width, new_height))
                    self.frames.append(processed_frame)
                else:
                    print(f"Animation frame not found: {path}")
            except pygame.error as e:
                print(f"Error loading animation frame '{path}': {e}")
        # If frames loaded and no image exists, set first frame as image
        if self.frames and not self.image:
            self.set_image(self.frames[0])
    
    def play_animation(self, loop=True):
        """
        Start playing the animation
        
        Args:
            loop: Whether to loop the animation
        """
        if self.frames:
            self.is_animating = True
            self.animation_loop = loop
            self.current_frame = 0
            self.image = self.frames[self.current_frame]
            old_topleft = self.rect.topleft if hasattr(self, 'rect') else self.position
            self.rect = self.image.get_rect()
            self.rect.topleft = old_topleft
    
    def stop_animation(self):
        """Stop the animation and revert to the original image"""
        self.is_animating = False
        if hasattr(self, 'original_image'):
            self.image = self.original_image.copy()
            old_topleft = self.rect.topleft
            self.rect = self.image.get_rect()
            self.rect.topleft = old_topleft
    
    def update(self, dt):
        """
        Update the sprite animation
        
        Args:
            dt: Time delta in seconds
        """
        if self.is_animating and self.frames:
            self.animation_timer += dt
            if self.animation_timer >= self.animation_speed:
                self.animation_timer = 0
                self.current_frame = (self.current_frame + 1) % len(self.frames)
                if self.current_frame == 0 and not self.animation_loop:
                    self.stop_animation()
                else:
                    self.image = self.frames[self.current_frame]
                    old_topleft = self.rect.topleft
                    self.rect = self.image.get_rect()
                    self.rect.topleft = old_topleft
    
    def draw(self, screen):
        """
        Draw the sprite on the screen
        
        Args:
            screen: Pygame surface to draw on
        """
        if self.visible and hasattr(self, 'image') and hasattr(self, 'rect'):
            screen.blit(self.image, self.rect)