"""
Save Manager Module
Handles game saves, loads, and save file management
"""
import os
import json
import time
import shutil
from datetime import datetime

class SaveManager:
    """
    Manages game save files, allowing multiple save slots and autosaving
    """
    # Maximum number of save slots
    MAX_SAVE_SLOTS = 5
    
    # Maximum number of autosaves to keep
    MAX_AUTOSAVES = 3
    
    # Save directory relative to the game directory
    SAVE_DIR = os.path.join("game", "data", "saves")
    
    def __init__(self, game_manager):
        """Initialize the save manager"""
        self.game_manager = game_manager
        self.save_slots = []
        self.autosave_enabled = True
        self.autosave_interval = 600  # 10 minutes
        self.last_autosave_time = time.time()
        
        # Create save directory if it doesn't exist
        os.makedirs(self.SAVE_DIR, exist_ok=True)
        
        # Load save slot metadata
        self._refresh_save_slots()
    
    def _refresh_save_slots(self):
        """Refresh the list of available save slots"""
        self.save_slots = []
        
        # Get all save files
        for filename in os.listdir(self.SAVE_DIR):
            if filename.endswith(".json") and not filename.startswith("autosave"):
                try:
                    # Get save slot number
                    slot_num = int(filename.split("_")[1].split(".")[0])
                    
                    # Read save metadata without loading the entire save
                    save_path = os.path.join(self.SAVE_DIR, filename)
                    metadata = self._get_save_metadata(save_path)
                    
                    self.save_slots.append({
                        "slot": slot_num,
                        "filename": filename,
                        "path": save_path,
                        "metadata": metadata
                    })
                except (ValueError, IndexError, KeyError) as e:
                    print(f"Error parsing save file {filename}: {e}")
        
        # Sort slots by slot number
        self.save_slots.sort(key=lambda x: x["slot"])
    
    def _get_save_metadata(self, save_path):
        """Extract metadata from a save file without loading all data"""
        try:
            with open(save_path, 'r') as save_file:
                data = json.load(save_file)
                
                # Extract basic metadata
                return {
                    "version": data.get("version", "Unknown"),
                    "save_time": data.get("save_time", 0),
                    "save_date": datetime.fromtimestamp(data.get("save_time", 0)).strftime("%Y-%m-%d %H:%M"),
                    "player_level": data.get("player_data", {}).get("level", 1),
                    "player_name": data.get("player_data", {}).get("name", "Unknown"),
                    "missions_completed": len(data.get("player_data", {}).get("missions_completed", [])),
                    "screenshot": data.get("screenshot", None)
                }
        except Exception as e:
            print(f"Error reading save metadata: {e}")
            return {
                "version": "Error",
                "save_time": 0,
                "save_date": "Unknown",
                "player_level": 0,
                "player_name": "Error",
                "missions_completed": 0,
                "screenshot": None
            }
    
    def save_game(self, slot=0, screenshot=None, is_autosave=False):
        """
        Save the current game state to the specified slot
        
        Args:
            slot (int): Save slot number (0-4, 0 is default)
            screenshot (Surface): Optional pygame surface to store as thumbnail
            is_autosave (bool): Whether this is an autosave
            
        Returns:
            bool: Whether the save was successful
        """
        # Generate filename based on slot and autosave status
        if is_autosave:
            filename = f"autosave_{int(time.time())}.json"
        else:
            filename = f"save_{slot}.json"
        
        save_path = os.path.join(self.SAVE_DIR, filename)
        
        # Gather save data from game manager
        save_data = {
            "version": self.game_manager.version,
            "player_data": self.game_manager.player_data,
            "statistics": self.game_manager.statistics,
            "achievements": self.game_manager.achievements,
            "settings": self.game_manager.settings,
            "save_time": time.time(),
            "save_type": "autosave" if is_autosave else "manual",
            "save_slot": slot
        }
        
        # Add screenshot if provided
        if screenshot:
            # Store screenshot path (screenshots saved separately)
            screenshot_dir = os.path.join(self.SAVE_DIR, "screenshots")
            os.makedirs(screenshot_dir, exist_ok=True)
            screenshot_filename = f"screenshot_{int(time.time())}.png"
            screenshot_path = os.path.join(screenshot_dir, screenshot_filename)
            
            # Save screenshot
            try:
                import pygame
                pygame.image.save(screenshot, screenshot_path)
                save_data["screenshot"] = screenshot_path
            except Exception as e:
                print(f"Error saving screenshot: {e}")
        
        # Save the data
        try:
            with open(save_path, 'w') as save_file:
                json.dump(save_data, save_file, indent=2)
            
            print(f"Game saved successfully to {save_path}")
            
            # Clean up old autosaves if needed
            if is_autosave:
                self._cleanup_autosaves()
            
            # Update save slots
            self._refresh_save_slots()
            
            self.game_manager.statistics["last_save_time"] = time.time()
            self.last_autosave_time = time.time()
            return True
            
        except Exception as e:
            print(f"Error saving game: {e}")
            return False
    
    def load_game(self, slot=0, autosave_index=None):
        """
        Load a saved game from the specified slot
        
        Args:
            slot (int): Save slot to load (0-4, 0 is default)
            autosave_index (int): Index of autosave to load (ignores slot if provided)
            
        Returns:
            bool: Whether the load was successful
        """
        # Determine which save file to load
        if autosave_index is not None:
            # Load from autosave
            autosaves = self._get_autosaves()
            if not autosaves or autosave_index >= len(autosaves):
                print(f"No autosave at index {autosave_index}")
                return False
            
            save_path = autosaves[autosave_index]["path"]
        else:
            # Load from regular save slot
            save_path = os.path.join(self.SAVE_DIR, f"save_{slot}.json")
            if not os.path.exists(save_path):
                print(f"No save file found in slot {slot}")
                return False
        
        # Load the save file
        try:
            with open(save_path, 'r') as save_file:
                save_data = json.load(save_file)
                
            # Check version compatibility
            if "version" in save_data:
                saved_version = save_data["version"]
                # Simple version check - could be more sophisticated
                if saved_version != self.game_manager.version:
                    print(f"Warning: Save file version ({saved_version}) differs from game version ({self.game_manager.version})")
            
            # Load saved data
            if "player_data" in save_data:
                self.game_manager.player_data = save_data["player_data"]
            
            if "statistics" in save_data:
                self.game_manager.statistics = save_data["statistics"]
                # Reset session start time
                self.game_manager.statistics["session_start_time"] = time.time()
            
            if "achievements" in save_data:
                self.game_manager.achievements = save_data["achievements"]
                
            if "settings" in save_data:
                self.game_manager.settings = save_data["settings"]
                
            print(f"Game loaded successfully from {save_path}")
            return True
            
        except Exception as e:
            print(f"Error loading game: {e}")
            return False
    
    def delete_save(self, slot=0, autosave_index=None):
        """
        Delete a saved game
        
        Args:
            slot (int): Save slot to delete (0-4, 0 is default)
            autosave_index (int): Index of autosave to delete (ignores slot if provided)
            
        Returns:
            bool: Whether the deletion was successful
        """
        # Determine which save file to delete
        if autosave_index is not None:
            # Delete an autosave
            autosaves = self._get_autosaves()
            if not autosaves or autosave_index >= len(autosaves):
                print(f"No autosave at index {autosave_index}")
                return False
            
            save_path = autosaves[autosave_index]["path"]
        else:
            # Delete from regular save slot
            save_path = os.path.join(self.SAVE_DIR, f"save_{slot}.json")
            if not os.path.exists(save_path):
                print(f"No save file found in slot {slot}")
                return False
        
        # Delete the save file
        try:
            # Also delete the screenshot if it exists
            try:
                with open(save_path, 'r') as save_file:
                    save_data = json.load(save_file)
                    
                if "screenshot" in save_data and save_data["screenshot"]:
                    screenshot_path = save_data["screenshot"]
                    if os.path.exists(screenshot_path):
                        os.remove(screenshot_path)
            except:
                pass
            
            # Delete the save file
            os.remove(save_path)
            print(f"Save file {save_path} deleted successfully")
            
            # Update save slots
            self._refresh_save_slots()
            return True
            
        except Exception as e:
            print(f"Error deleting save file: {e}")
            return False
    
    def check_autosave(self):
        """
        Check if it's time to autosave, and perform autosave if needed
        
        Returns:
            bool: Whether an autosave was performed
        """
        if not self.autosave_enabled:
            return False
            
        current_time = time.time()
        if current_time - self.last_autosave_time >= self.autosave_interval:
            print("Performing autosave...")
            return self.save_game(is_autosave=True)
            
        return False
    
    def _cleanup_autosaves(self):
        """Clean up old autosaves, keeping only the most recent ones"""
        autosaves = self._get_autosaves()
        
        # If we have more than the maximum, delete the oldest ones
        if len(autosaves) > self.MAX_AUTOSAVES:
            # Sort by save time, oldest first
            autosaves.sort(key=lambda x: x["metadata"]["save_time"])
            
            # Delete the oldest autosaves
            for i in range(len(autosaves) - self.MAX_AUTOSAVES):
                try:
                    os.remove(autosaves[i]["path"])
                    print(f"Deleted old autosave: {autosaves[i]['filename']}")
                except Exception as e:
                    print(f"Error deleting old autosave: {e}")
    
    def _get_autosaves(self):
        """Get a list of all autosaves"""
        autosaves = []
        
        for filename in os.listdir(self.SAVE_DIR):
            if filename.startswith("autosave_") and filename.endswith(".json"):
                save_path = os.path.join(self.SAVE_DIR, filename)
                metadata = self._get_save_metadata(save_path)
                
                autosaves.append({
                    "filename": filename,
                    "path": save_path,
                    "metadata": metadata
                })
        
        # Sort by save time, newest first
        autosaves.sort(key=lambda x: x["metadata"]["save_time"], reverse=True)
        
        return autosaves
    
    def get_next_empty_slot(self):
        """Find the next empty save slot"""
        for slot in range(self.MAX_SAVE_SLOTS):
            save_path = os.path.join(self.SAVE_DIR, f"save_{slot}.json")
            if not os.path.exists(save_path):
                return slot
        
        # If all slots are used, recommend the oldest one
        oldest_slot = 0
        oldest_time = float('inf')
        
        for i in range(self.MAX_SAVE_SLOTS):
            save_path = os.path.join(self.SAVE_DIR, f"save_{i}.json")
            try:
                with open(save_path, 'r') as save_file:
                    save_data = json.load(save_file)
                    save_time = save_data.get("save_time", 0)
                    
                    if save_time < oldest_time:
                        oldest_time = save_time
                        oldest_slot = i
            except:
                pass
        
        return oldest_slot
    
    def export_save(self, slot=0, export_path=None):
        """
        Export a save file to a user-specified location
        
        Args:
            slot (int): Save slot to export
            export_path (str): Path to export to (optional)
            
        Returns:
            str: Export path if successful, None otherwise
        """
        save_path = os.path.join(self.SAVE_DIR, f"save_{slot}.json")
        if not os.path.exists(save_path):
            print(f"No save file found in slot {slot}")
            return None
        
        # Default export location
        if not export_path:
            # Get Documents folder or default to home directory
            user_home = os.path.expanduser("~")
            documents_dir = os.path.join(user_home, "Documents")
            if not os.path.exists(documents_dir):
                documents_dir = user_home
            
            # Create export filename with timestamp
            timestamp = datetime.now().strftime("%Y%m%d_%H%M%S")
            export_path = os.path.join(documents_dir, f"WH40K_save_{timestamp}.json")
        
        try:
            # Copy the save file
            shutil.copy2(save_path, export_path)
            print(f"Save file exported to {export_path}")
            return export_path
        except Exception as e:
            print(f"Error exporting save file: {e}")
            return None
    
    def import_save(self, import_path, slot=None):
        """
        Import a save file from a user-specified location
        
        Args:
            import_path (str): Path to import from
            slot (int): Save slot to import to (optional, auto-selects if None)
            
        Returns:
            bool: Whether the import was successful
        """
        if not os.path.exists(import_path):
            print(f"Import file not found: {import_path}")
            return False
        
        # Validate that this is a proper save file
        try:
            with open(import_path, 'r') as import_file:
                save_data = json.load(import_file)
                
                # Simple validation
                required_keys = ["version", "player_data", "statistics", "settings"]
                for key in required_keys:
                    if key not in save_data:
                        print(f"Invalid save file: missing {key}")
                        return False
        except:
            print("Invalid save file format")
            return False
        
        # Determine which slot to use
        if slot is None:
            slot = self.get_next_empty_slot()
        
        # Import the save file
        save_path = os.path.join(self.SAVE_DIR, f"save_{slot}.json")
        try:
            shutil.copy2(import_path, save_path)
            print(f"Save file imported to slot {slot}")
            
            # Update save slots
            self._refresh_save_slots()
            return True
        except Exception as e:
            print(f"Error importing save file: {e}")
            return False 