"""
Resource Manager Module
Handles loading and managing game assets
"""
import os
import pygame
from game.core.sprite import Sprite

class ResourceManager:
    """
    Loads and manages game resources such as images, sounds, and fonts
    """
    def __init__(self):
        """Initialize the resource manager"""
        self.images = {}
        self.sprites = {}
        self.animations = {}
        self.sounds = {}
        self.fonts = {}
        
        # Base directories
        self.base_dir = os.path.join("game")
        self.image_dir = os.path.join(self.base_dir, "assets", "images")
        self.sound_dir = os.path.join(self.base_dir, "assets", "sounds")
        
        # Initialize resources
        self._init_fonts()
    
    def _init_fonts(self):
        """Initialize game fonts"""
        pygame.font.init()
        
        # Create common fonts
        self.fonts["small"] = pygame.font.Font(None, 24)
        self.fonts["medium"] = pygame.font.Font(None, 36)
        self.fonts["large"] = pygame.font.Font(None, 48)
        self.fonts["title"] = pygame.font.Font(None, 72)
    
    def load_image(self, name, file_path, color_key=None):
        """
        Load an image into the resource manager
        
        Args:
            name: Identifier for the image
            file_path: Path to the image file (relative to image_dir)
            color_key: Transparent color (if any)
        """
        if name in self.images:
            return self.images[name]
        
        fullpath = os.path.join(self.image_dir, file_path)
        try:
            image = pygame.image.load(fullpath)
            if color_key is not None:
                image = image.convert()
                if color_key == -1:
                    color_key = image.get_at((0, 0))
                image.set_colorkey(color_key, pygame.RLEACCEL)
            else:
                image = image.convert_alpha()
            
            self.images[name] = image
            return image
        except pygame.error as e:
            print(f"Cannot load image: {fullpath}")
            print(f"Error: {e}")
            # Return a placeholder image (red square)
            placeholder = pygame.Surface((64, 64))
            placeholder.fill((255, 0, 0))
            return placeholder
    
    def load_sprite(self, name, file_path, position=(0, 0), scale=1.0, color_key=None):
        """
        Load an image as a sprite
        
        Args:
            name: Identifier for the sprite
            file_path: Path to the image file (relative to image_dir)
            position: Initial position for the sprite
            scale: Scale factor for the sprite
            color_key: Transparent color (if any)
        """
        if name in self.sprites:
            return self.sprites[name]
            
        fullpath = os.path.join(self.image_dir, file_path)
        
        # Create a new sprite
        sprite = Sprite(fullpath, position, scale)
        self.sprites[name] = sprite
        return sprite
    
    def load_animation(self, name, file_paths, frame_duration=0.1):
        """
        Load an animation sequence
        
        Args:
            name: Identifier for the animation
            file_paths: List of paths to frame images (relative to image_dir)
            frame_duration: Duration of each frame in seconds
        """
        if name in self.animations:
            return self.animations[name]
            
        frames = []
        for path in file_paths:
            fullpath = os.path.join(self.image_dir, path)
            try:
                if os.path.exists(fullpath):
                    frame = pygame.image.load(fullpath).convert_alpha()
                    frames.append(frame)
                else:
                    print(f"Animation frame not found: {fullpath}")
            except pygame.error as e:
                print(f"Error loading animation frame: {e}")
                
        if frames:
            self.animations[name] = {
                "frames": frames,
                "frame_duration": frame_duration
            }
            return self.animations[name]
        else:
            print(f"No valid frames loaded for animation: {name}")
            return None
    
    def load_sound(self, name, file_path):
        """
        Load a sound effect into the resource manager
        
        Args:
            name: Identifier for the sound
            file_path: Path to the sound file (relative to sound_dir)
        """
        if name in self.sounds:
            return self.sounds[name]
        
        fullpath = os.path.join(self.sound_dir, file_path)
        try:
            sound = pygame.mixer.Sound(fullpath)
            self.sounds[name] = sound
            return sound
        except pygame.error as e:
            print(f"Cannot load sound: {fullpath}")
            print(f"Error: {e}")
            # Return a dummy sound object
            return pygame.mixer.Sound(buffer=bytearray(44100))  # 1 second of silence
    
    def get_image(self, name):
        """Get a loaded image by name"""
        if name in self.images:
            return self.images[name]
        print(f"Warning: Image '{name}' not found in resource manager")
        # Return a placeholder image (purple square - indicating missing texture)
        placeholder = pygame.Surface((64, 64))
        placeholder.fill((128, 0, 128))
        return placeholder
    
    def get_sprite(self, name):
        """Get a loaded sprite by name"""
        if name in self.sprites:
            return self.sprites[name]
        print(f"Warning: Sprite '{name}' not found in resource manager")
        # Return a new placeholder sprite
        placeholder_sprite = Sprite()
        return placeholder_sprite
    
    def get_animation(self, name):
        """Get a loaded animation by name"""
        if name in self.animations:
            return self.animations[name]
        print(f"Warning: Animation '{name}' not found in resource manager")
        return None
    
    def get_sound(self, name):
        """Get a loaded sound by name"""
        if name in self.sounds:
            return self.sounds[name]
        print(f"Warning: Sound '{name}' not found in resource manager")
        # Return a dummy sound object
        return pygame.mixer.Sound(buffer=bytearray(44100))  # 1 second of silence
    
    def get_font(self, name):
        """Get a loaded font by name"""
        if name in self.fonts:
            return self.fonts[name]
        print(f"Warning: Font '{name}' not found in resource manager")
        # Return the default font
        return pygame.font.Font(None, 36)
    
    def load_unit_sprites(self, faction):
        """
        Load all unit sprites for a specific faction
        
        Args:
            faction: Faction name (e.g., "space_marines")
        """
        faction_dir = os.path.join(self.image_dir, "units", faction)
        if not os.path.exists(faction_dir):
            print(f"Faction directory not found: {faction_dir}")
            return
            
        for filename in os.listdir(faction_dir):
            if filename.endswith(".png") or filename.endswith(".jpg"):
                # Remove file extension
                name = os.path.splitext(filename)[0]
                sprite_name = f"{faction}_{name}"
                file_path = os.path.join("units", faction, filename)
                
                # Load as both image and sprite
                self.load_image(sprite_name, file_path)
                self.load_sprite(sprite_name, file_path)
                
                print(f"Loaded unit sprite: {sprite_name}")
    
    def load_all_unit_sprites(self):
        """Load all unit sprites for all factions"""
        units_dir = os.path.join(self.image_dir, "units")
        if not os.path.exists(units_dir):
            print(f"Units directory not found: {units_dir}")
            return
            
        for faction in os.listdir(units_dir):
            faction_dir = os.path.join(units_dir, faction)
            if os.path.isdir(faction_dir):
                self.load_unit_sprites(faction)
    
    def load_ui_elements(self):
        """Load UI elements"""
        ui_dir = os.path.join(self.image_dir, "ui")
        if not os.path.exists(ui_dir):
            print(f"UI directory not found: {ui_dir}")
            return
            
        for filename in os.listdir(ui_dir):
            if filename.endswith(".png") or filename.endswith(".jpg"):
                # Remove file extension
                name = os.path.splitext(filename)[0]
                ui_name = f"ui_{name}"
                file_path = os.path.join("ui", filename)
                
                # Load as image
                self.load_image(ui_name, file_path)
                print(f"Loaded UI element: {ui_name}")
    
    def load_backgrounds(self):
        """Load background images"""
        bg_dir = os.path.join(self.image_dir, "backgrounds")
        if not os.path.exists(bg_dir):
            print(f"Backgrounds directory not found: {bg_dir}")
            return
            
        for filename in os.listdir(bg_dir):
            if filename.endswith(".png") or filename.endswith(".jpg"):
                # Remove file extension
                name = os.path.splitext(filename)[0]
                bg_name = f"bg_{name}"
                file_path = os.path.join("backgrounds", filename)
                
                # Load as image
                self.load_image(bg_name, file_path)
                print(f"Loaded background: {bg_name}")
    
    def preload_assets(self):
        """Preload commonly used assets"""
        print("Preloading game assets...")
        
        # Preload unit sprites
        self.load_all_unit_sprites()
        
        # Preload UI elements
        self.load_ui_elements()
        
        # Preload backgrounds
        self.load_backgrounds()
        
        print("Asset preloading complete!") 