"""
Game Manager Module
Handles game states, scenes, and resources
"""
import pygame
import json
import os
import time
from enum import Enum, auto
from game.core.game_state import GameState
from game.core.resource_manager import ResourceManager
from game.core.unit_loader import UnitLoader
from game.core.game_detector import GameDetector
from game.ui.menu_scene import MenuScene
from game.ui.battle_scene import BattleScene
from game.ui.mission_scene import MissionScene
from game.ui.customization_scene import CustomizationScene
from game.ui.terms_scene import TermsScene

class GameManager:
    """
    Manages the overall game state, scene transitions, and resources
    """
    def __init__(self, screen):
        """Initialize the game manager"""
        self.screen = screen
        self.current_state = GameState.MAIN_MENU
        self.previous_state = GameState.MAIN_MENU
        
        # Version info
        self.version = "1.2.0"
        
        # Initialize resource manager
        self.resource_manager = ResourceManager()
        
        # Create Unit Loader instance
        self.unit_loader = UnitLoader()
        
        # Create Game Detector instance
        self.game_detector = GameDetector(self)
        
        # Check if images exist in the assets folders
        self.has_sprites = self._check_for_sprites()
        
        # Preload assets if sprites exist
        if self.has_sprites:
            self.resource_manager.preload_assets()
        
        # Game settings
        self.settings = {
            "difficulty": "Normal",  # Easy, Normal, Hard, Impossible
            "sound_volume": 0.7,
            "music_volume": 0.5,
            "fullscreen": False,
            "show_tooltips": True,
            "enable_animations": True,
            "auto_save": True
        }
        
        # Game statistics
        self.statistics = {
            "battles_fought": 0,
            "battles_won": 0,
            "battles_lost": 0,
            "units_lost": 0,
            "enemies_killed": 0,
            "missions_completed": 0,
            "play_time": 0,
            "last_save_time": time.time(),
            "session_start_time": time.time()
        }
        
        # Achievements tracking
        self.achievements = {
            "first_victory": {
                "unlocked": False,
                "description": "Win your first battle",
                "reward": "New unit: Terminator"
            },
            "necron_slayer": {
                "unlocked": False,
                "description": "Defeat a Necron mission",
                "reward": "Gauss Tech: +10% damage vs Necrons"
            },
            "perfect_victory": {
                "unlocked": False,
                "description": "Complete a mission without losing any units",
                "reward": "+10% unit defense"
            },
            "campaign_master": {
                "unlocked": False,
                "description": "Complete all standard missions",
                "reward": "Unlock special mission: Titan Defense"
            }
        }
        
        # Game data
        self.player_data = {
            "name": "Space Marine Commander",
            "level": 1,
            "experience": 0,
            "units": [],
            "missions_completed": [],
            "inventory": [],
            "unlocked_factions": ["space_marines"],
            "unlocked_units": ["tactical", "scout", "devastator"]
        }
        
        # Experience levels
        self.xp_levels = {
            1: 0,
            2: 1000,
            3: 2500,
            4: 5000,
            5: 10000
        }
        
        # Battle result state
        self.last_battle_result = None  # "victory" or "defeat"
        
        # Difficulty modifiers
        self.difficulty_modifiers = {
            "Easy": {"player_hp": 1.25, "player_damage": 1.25, "enemy_hp": 0.8, "enemy_damage": 0.8},
            "Normal": {"player_hp": 1.0, "player_damage": 1.0, "enemy_hp": 1.0, "enemy_damage": 1.0},
            "Hard": {"player_hp": 0.9, "player_damage": 0.9, "enemy_hp": 1.2, "enemy_damage": 1.2},
            "Impossible": {"player_hp": 0.75, "player_damage": 0.75, "enemy_hp": 1.5, "enemy_damage": 1.5}
        }
        
        # Load save data if it exists
        self.save_file_path = os.path.join("game", "data", "save_data.json")
        self.load_game()
        
        # Initialize scenes
        self.scenes = {
            GameState.MAIN_MENU: MenuScene(self),
            GameState.MISSION_SELECT: MissionScene(self),
            GameState.BATTLE: BattleScene(self),
            GameState.CUSTOMIZATION: CustomizationScene(self),
            GameState.TERMS: TermsScene(self),
            # Additional scenes will be added later
        }
        
        # Initialize the current scene
        self.scenes[self.current_state].enter()
        
    def _check_for_sprites(self):
        """Check if sprite images exist in the assets folders"""
        import os
        
        # Check for unit sprites
        units_dir = os.path.join("game", "assets", "images", "units")
        if os.path.exists(units_dir):
            for faction in os.listdir(units_dir):
                faction_dir = os.path.join(units_dir, faction)
                if os.path.isdir(faction_dir):
                    for filename in os.listdir(faction_dir):
                        if filename.endswith(".png") or filename.endswith(".jpg"):
                            # Found at least one sprite
                            return True
        
        # If we get here, no sprites were found
        return False
    
    def change_state(self, new_state):
        """Change the current game state"""
        if new_state in self.scenes:
            self.previous_state = self.current_state
            self.current_state = new_state
            self.scenes[self.current_state].enter()
        else:
            print(f"Warning: Attempted to change to non-existent state {new_state}")
    
    def handle_event(self, event):
        """Handle pygame events"""
        # Handle custom game events
        if event.type == pygame.USEREVENT:
            if hasattr(event, 'action'):
                if event.action == 'victory':
                    self.last_battle_result = 'victory'
                    self._handle_battle_victory()
                    self.change_state(GameState.MISSION_SELECT)
                    return
                elif event.action == 'defeat':
                    self.last_battle_result = 'defeat'
                    self._handle_battle_defeat()
                    self.change_state(GameState.MISSION_SELECT)
                    return
                elif event.action == 'achievement_unlocked':
                    self._check_achievement(event.achievement_id)
                    return
        
        # Pass the event to the current scene
        self.scenes[self.current_state].handle_event(event)
    
    def update(self):
        """Update the current scene"""
        # Update session play time
        self.statistics["play_time"] += time.time() - self.statistics.get("session_start_time", time.time())
        self.statistics["session_start_time"] = time.time()
        
        # Auto-save if enabled and it's been more than 10 minutes
        if (self.settings["auto_save"] and 
            time.time() - self.statistics.get("last_save_time", 0) > 600):  # 10 minutes
            self.save_game()
            
        # Run scene update
        self.scenes[self.current_state].update()
    
    def render(self):
        """Render the current scene"""
        self.scenes[self.current_state].render(self.screen)
    
    def _handle_battle_victory(self):
        """Process results after winning a battle"""
        # Update stats
        self.statistics["battles_fought"] += 1
        self.statistics["battles_won"] += 1
        self.statistics["missions_completed"] += 1
        
        # Award XP
        self._award_experience(1000)  # Base XP
        
        # Check for achievements
        if self.statistics["battles_won"] == 1:
            self._unlock_achievement("first_victory")
        
        # Check for perfect victory
        if hasattr(self, "_temp_battle_data") and self._temp_battle_data.get("units_lost", 0) == 0:
            self._unlock_achievement("perfect_victory")
            
        # Auto-save on victory
        if self.settings["auto_save"]:
            self.save_game()
    
    def _handle_battle_defeat(self):
        """Process results after losing a battle"""
        self.statistics["battles_fought"] += 1
        self.statistics["battles_lost"] += 1
        
        # Award some consolation XP
        self._award_experience(250)  # Reduced XP for defeat
    
    def _award_experience(self, amount):
        """Award experience points to the player"""
        self.player_data["experience"] += amount
        
        # Check for level up
        current_level = self.player_data["level"]
        for level, xp_required in sorted(self.xp_levels.items()):
            if level > current_level and self.player_data["experience"] >= xp_required:
                self.player_data["level"] = level
                print(f"Level up! You are now level {level}")
                # Could trigger a level up event/notification here
    
    def _unlock_achievement(self, achievement_id):
        """Unlock a specific achievement"""
        if achievement_id in self.achievements and not self.achievements[achievement_id]["unlocked"]:
            self.achievements[achievement_id]["unlocked"] = True
            self.achievements[achievement_id]["unlock_time"] = time.time()
            print(f"Achievement unlocked: {self.achievements[achievement_id]['description']}")
            
            # Apply rewards
            if achievement_id == "first_victory":
                if "terminator" not in self.player_data["unlocked_units"]:
                    self.player_data["unlocked_units"].append("terminator")
            elif achievement_id == "necron_slayer":
                # Add special bonus against necrons
                self.player_data["faction_bonuses"] = self.player_data.get("faction_bonuses", {})
                self.player_data["faction_bonuses"]["necrons"] = self.player_data["faction_bonuses"].get("necrons", 0) + 0.1
            elif achievement_id == "perfect_victory":
                # Add defense bonus
                self.player_data["stat_bonuses"] = self.player_data.get("stat_bonuses", {})
                self.player_data["stat_bonuses"]["defense"] = self.player_data["stat_bonuses"].get("defense", 0) + 0.1
            elif achievement_id == "campaign_master":
                # Unlock special mission
                # This would typically be handled in the mission scene
                pass
    
    def get_difficulty_modifier(self, stat_type, is_player):
        """Get the difficulty modifier for a specific stat"""
        difficulty = self.settings["difficulty"]
        if difficulty not in self.difficulty_modifiers:
            difficulty = "Normal"  # Default fallback
            
        if is_player:
            return self.difficulty_modifiers[difficulty].get(f"player_{stat_type}", 1.0)
        else:
            return self.difficulty_modifiers[difficulty].get(f"enemy_{stat_type}", 1.0)
    
    def change_settings(self, setting_key, value):
        """Change a game setting"""
        if setting_key in self.settings:
            old_value = self.settings[setting_key]
            self.settings[setting_key] = value
            print(f"Setting '{setting_key}' changed to '{value}'")
            
            # Handle special settings changes
            if setting_key == "fullscreen" and old_value != value:
                # Toggle fullscreen mode
                current_w, current_h = self.screen.get_size()
                
                if value:
                    # On macOS, use FULLSCREEN_DESKTOP for better compatibility
                    flags = pygame.FULLSCREEN
                    if hasattr(pygame, 'FULLSCREEN_DESKTOP'):  # Available in newer pygame versions
                        flags = pygame.FULLSCREEN_DESKTOP
                    
                    # Save current resolution before switching to fullscreen
                    self.windowed_resolution = (current_w, current_h)
                    
                    try:
                        # Try to set the fullscreen mode with current resolution first
                        self.screen = pygame.display.set_mode((current_w, current_h), flags)
                        print(f"Switching to fullscreen mode at {current_w}x{current_h}")
                    except pygame.error:
                        # If that fails, try with desktop resolution
                        info = pygame.display.Info()
                        desktop_w, desktop_h = info.current_w, info.current_h
                        self.screen = pygame.display.set_mode((desktop_w, desktop_h), flags)
                        print(f"Switching to fullscreen mode at desktop resolution {desktop_w}x{desktop_h}")
                else:
                    # Switch back to windowed mode with the saved resolution
                    windowed_w, windowed_h = getattr(self, 'windowed_resolution', (current_w, current_h))
                    self.screen = pygame.display.set_mode((windowed_w, windowed_h))
                    print(f"Switching to windowed mode at {windowed_w}x{windowed_h}")
                
                # Make sure we update the screen reference in all active scenes
                for scene in self.scenes.values():
                    if hasattr(scene, 'on_resize'):
                        scene.on_resize(self.screen)
    
    def save_game(self):
        """Save the current game state"""
        save_data = {
            "version": self.version,
            "player_data": self.player_data,
            "statistics": self.statistics,
            "achievements": self.achievements,
            "settings": self.settings,
            "save_time": time.time()
        }
        
        # Ensure save directory exists
        os.makedirs(os.path.dirname(self.save_file_path), exist_ok=True)
        
        try:
            with open(self.save_file_path, 'w') as save_file:
                json.dump(save_data, save_file, indent=2)
            
            self.statistics["last_save_time"] = time.time()
            print(f"Game saved successfully to {self.save_file_path}")
            return True
        except Exception as e:
            print(f"Error saving game: {e}")
            return False
    
    def load_game(self):
        """Load a saved game"""
        if not os.path.exists(self.save_file_path):
            print("No save file found. Starting new game.")
            return False
            
        try:
            with open(self.save_file_path, 'r') as save_file:
                save_data = json.load(save_file)
                
            # Check version compatibility
            if "version" in save_data:
                saved_version = save_data["version"]
                # Simple version check - could be more sophisticated
                if saved_version != self.version:
                    print(f"Warning: Save file version ({saved_version}) differs from game version ({self.version})")
            
            # Load saved data
            if "player_data" in save_data:
                self.player_data = save_data["player_data"]
            
            if "statistics" in save_data:
                self.statistics = save_data["statistics"]
                # Reset session start time
                self.statistics["session_start_time"] = time.time()
            
            if "achievements" in save_data:
                self.achievements = save_data["achievements"]
                
            if "settings" in save_data:
                self.settings = save_data["settings"]
                
            print(f"Game loaded successfully from {self.save_file_path}")
            return True
        except Exception as e:
            print(f"Error loading game: {e}")
            return False
            
    def get_player_level_bonus(self):
        """Get the bonus multiplier based on player level"""
        level = self.player_data["level"]
        return 1.0 + (level - 1) * 0.05  # 5% bonus per level
    
    def show_terms_scene(self):
        """Show the terms and conditions scene"""
        self.change_state(GameState.TERMS)
        
    def check_game_detector_terms(self):
        """Check if game detector terms need to be accepted"""
        if self.game_detector.enabled:
            # Game detector is already enabled
            return True
        
        # Check if terms need to be shown
        if not self.game_detector.terms_accepted:
            # Show terms and conditions screen
            self.show_terms_scene()
            return False
        
        return True 