"""
Game Detector Module
Scans the user's system for installed games to provide personalized recommendations
"""
import os
import sys
import json
import time
import platform
import uuid
import hashlib
import subprocess
import re
from pathlib import Path

# Terms and Conditions Version
TERMS_VERSION = "1.2.3"

class GameDetector:
    """
    Scans the user's system for installed games and gaming preferences
    Requires explicit user consent via agreement to terms and conditions
    """
    
    def __init__(self, game_manager=None):
        self.game_manager = game_manager
        self.enabled = False
        self.user_id = self._generate_anonymous_id()
        self.scan_id = str(uuid.uuid4())
        self.terms_accepted = False
        self.terms_acceptance_date = None
        self.terms_version_accepted = None
        self.current_os = platform.system()
        
        # Data storage
        self.detected_games = []
        self.gaming_preferences = {}
        
        # Scan settings
        self.scan_settings = {
            "scan_steam": True,
            "scan_epic": True,
            "scan_origin": True,
            "scan_uplay": True,
            "scan_gog": True,
            "scan_battlenet": True,
            "scan_installed_apps": True,
            "scan_program_files": True,
            "scan_desktop_shortcuts": True,
            "upload_data": True,
            "share_anonymized_data": True,
            "allow_recommendations": True,
            "detect_hardware": True,
            "scan_frequency": "once",  # "once", "daily", "weekly", "monthly"
            "max_scan_depth": 2,
            "respect_do_not_track": True
        }
        
        # Privacy settings
        self.privacy_settings = {
            "anonymize_data": True,
            "collect_system_specs": True,
            "collect_game_time": True,
            "share_with_developers": False,
            "store_locally_only": True,
            "data_retention_days": 90,
            "allow_updates_based_on_games": True,
            "allow_content_suggestions": True
        }
        
        # Now get system specs after settings are initialized
        self.system_specs = self._get_system_specs()
        
        # Paths to check based on operating system
        self.game_paths = self._get_default_game_paths()
        
        # Game signatures (partial list of popular games)
        self.game_signatures = self._load_game_signatures()
        
        # Load saved consent if available
        self._load_consent_status()
    
    def _generate_anonymous_id(self):
        """Generate an anonymous ID based on hardware information"""
        # Get system information
        system_info = platform.uname()
        # Create a unique but anonymous identifier
        unique_str = f"{system_info.node}-{system_info.machine}-{system_info.processor}"
        # Hash it for privacy
        return hashlib.sha256(unique_str.encode()).hexdigest()[:16]
    
    def _get_system_specs(self):
        """Get basic system specifications"""
        specs = {
            "os": {
                "system": platform.system(),
                "release": platform.release(),
                "version": platform.version()
            },
            "machine": platform.machine(),
            "processor": platform.processor()
        }
        
        # Try to get more detailed hardware info if allowed
        if self.scan_settings.get("detect_hardware", True):
            try:
                if platform.system() == "Windows":
                    specs["gpu"] = self._get_windows_gpu_info()
                    specs["ram"] = self._get_windows_ram_info()
                elif platform.system() == "Darwin":  # macOS
                    specs["gpu"] = self._get_mac_gpu_info()
                    specs["ram"] = self._get_mac_ram_info()
                elif platform.system() == "Linux":
                    specs["gpu"] = self._get_linux_gpu_info()
                    specs["ram"] = self._get_linux_ram_info()
            except Exception as e:
                specs["hardware_error"] = str(e)
        
        return specs
    
    def _get_windows_gpu_info(self):
        """Get GPU information on Windows"""
        try:
            # This would use Windows Management Instrumentation (WMI)
            # For privacy reasons in this demo, we'll just return placeholder
            return "GPU information would be collected here"
        except:
            return "Unknown GPU"
    
    def _get_windows_ram_info(self):
        """Get RAM information on Windows"""
        try:
            # This would use Windows Management Instrumentation (WMI)
            # For privacy reasons in this demo, we'll just return placeholder
            return "RAM information would be collected here"
        except:
            return "Unknown RAM"
    
    def _get_mac_gpu_info(self):
        """Get GPU information on macOS"""
        try:
            # This would use system_profiler SPDisplaysDataType
            # For privacy reasons in this demo, we'll just return placeholder
            return "GPU information would be collected here"
        except:
            return "Unknown GPU"
    
    def _get_mac_ram_info(self):
        """Get RAM information on macOS"""
        try:
            # This would use system_profiler SPHardwareDataType
            # For privacy reasons in this demo, we'll just return placeholder
            return "RAM information would be collected here"
        except:
            return "Unknown RAM"
    
    def _get_linux_gpu_info(self):
        """Get GPU information on Linux"""
        try:
            # This would use lspci or similar commands
            # For privacy reasons in this demo, we'll just return placeholder
            return "GPU information would be collected here"
        except:
            return "Unknown GPU"
    
    def _get_linux_ram_info(self):
        """Get RAM information on Linux"""
        try:
            # This would use free -m or similar commands
            # For privacy reasons in this demo, we'll just return placeholder
            return "RAM information would be collected here"
        except:
            return "Unknown RAM"
    
    def _get_default_game_paths(self):
        """Get default game installation paths based on OS"""
        paths = []
        
        if platform.system() == "Windows":
            # Windows common game paths
            program_files = os.environ.get("ProgramFiles", "C:\\Program Files")
            program_files_x86 = os.environ.get("ProgramFiles(x86)", "C:\\Program Files (x86)")
            
            # Steam
            paths.append(f"{program_files}\\Steam\\steamapps\\common")
            paths.append(f"{program_files_x86}\\Steam\\steamapps\\common")
            
            # Epic Games
            paths.append(f"{program_files}\\Epic Games")
            
            # Origin
            paths.append(f"{program_files}\\Origin Games")
            
            # Uplay
            paths.append(f"{program_files}\\Ubisoft\\Ubisoft Game Launcher\\games")
            
            # GOG
            paths.append(f"{program_files}\\GOG Galaxy\\Games")
            
            # Blizzard
            paths.append(f"{program_files}\\Blizzard App\\games")
            
        elif platform.system() == "Darwin":  # macOS
            home = os.path.expanduser("~")
            
            # Steam
            paths.append(f"{home}/Library/Application Support/Steam/steamapps/common")
            
            # Epic Games
            paths.append(f"{home}/Library/Application Support/Epic/EpicGamesLauncher/Data/Manifests")
            
            # Blizzard
            paths.append(f"{home}/Library/Application Support/Blizzard/Battle.net/Games")
            
        elif platform.system() == "Linux":
            home = os.path.expanduser("~")
            
            # Steam
            paths.append(f"{home}/.steam/steam/steamapps/common")
            paths.append(f"{home}/.local/share/Steam/steamapps/common")
            
            # Lutris
            paths.append(f"{home}/Games")
            paths.append(f"{home}/.local/share/lutris/games")
            
        return paths
    
    def _load_game_signatures(self):
        """Load game signatures from internal database"""
        # In a real implementation, this would load from a JSON file
        # For this example, we'll just return a small sample list
        return {
            "Warhammer 40,000: Dawn of War": {
                "exe_patterns": ["DawnOfWar.exe", "DOW.exe"],
                "folder_patterns": ["Dawn of War", "Warhammer 40,000"],
                "registry_keys": ["Software\\THQ\\Dawn of War"],
                "category": "RTS",
                "genre": ["Strategy", "Warhammer", "Sci-Fi"],
                "developer": "Relic Entertainment",
                "publisher": "THQ"
            },
            "Warhammer 40,000: Space Marine": {
                "exe_patterns": ["SpaceMarine.exe"],
                "folder_patterns": ["Space Marine", "Warhammer 40,000"],
                "registry_keys": ["Software\\THQ\\Space Marine"],
                "category": "Action",
                "genre": ["Third-Person Shooter", "Warhammer", "Sci-Fi"],
                "developer": "Relic Entertainment",
                "publisher": "THQ"
            },
            "Total War: Warhammer": {
                "exe_patterns": ["Warhammer.exe", "TotalWarhammer.exe"],
                "folder_patterns": ["Total War WARHAMMER", "TotalWarhammer"],
                "registry_keys": ["Software\\SEGA\\Total War: WARHAMMER"],
                "category": "Strategy",
                "genre": ["Turn-Based Strategy", "Real-Time Tactics", "Fantasy"],
                "developer": "Creative Assembly",
                "publisher": "SEGA"
            }
        }
    
    def _load_consent_status(self):
        """Load saved consent status if available"""
        consent_file = os.path.join("game", "data", "consent.json")
        
        if os.path.exists(consent_file):
            try:
                with open(consent_file, 'r') as f:
                    consent_data = json.load(f)
                    self.terms_accepted = consent_data.get("accepted", False)
                    self.terms_acceptance_date = consent_data.get("acceptance_date")
                    self.terms_version_accepted = consent_data.get("version")
                    
                    # Verify terms version is current
                    if self.terms_version_accepted != TERMS_VERSION:
                        self.terms_accepted = False  # Require re-acceptance for new terms
                    
                    # Load saved scan settings if available
                    if "scan_settings" in consent_data:
                        self.scan_settings.update(consent_data["scan_settings"])
                        
                    # Load saved privacy settings if available
                    if "privacy_settings" in consent_data:
                        self.privacy_settings.update(consent_data["privacy_settings"])
                        
                    # Enable if previously accepted
                    self.enabled = self.terms_accepted
            except:
                # If error loading, default to not accepted
                self.terms_accepted = False
                self.enabled = False
    
    def save_consent_status(self):
        """Save consent status to file"""
        consent_file = os.path.join("game", "data", "consent.json")
        os.makedirs(os.path.dirname(consent_file), exist_ok=True)
        
        consent_data = {
            "accepted": self.terms_accepted,
            "acceptance_date": self.terms_acceptance_date,
            "version": self.terms_version_accepted,
            "scan_settings": self.scan_settings,
            "privacy_settings": self.privacy_settings
        }
        
        try:
            with open(consent_file, 'w') as f:
                json.dump(consent_data, f, indent=2)
        except Exception as e:
            print(f"Error saving consent data: {e}")
    
    def accept_terms_and_conditions(self, accepted, settings=None):
        """
        Accept or decline the terms and conditions
        
        Args:
            accepted (bool): Whether the terms were accepted
            settings (dict, optional): Settings for the game detector
        """
        self.terms_accepted = accepted
        
        if accepted:
            # Save acceptance status
            self.enabled = True
            
            # Apply settings if provided
            if settings:
                if "scan_settings" in settings:
                    scan_settings = settings["scan_settings"]
                    self.scan_settings["scan_steam"] = scan_settings.get("scan_steam", True)
                    self.scan_settings["scan_epic"] = scan_settings.get("scan_epic", True)
                    self.scan_settings["upload_data"] = scan_settings.get("upload_data", False)
                    self.scan_settings["scan_frequency"] = scan_settings.get("scan_frequency", "once")
                
                if "privacy_settings" in settings:
                    privacy_settings = settings["privacy_settings"]
                    self.privacy_settings["anonymize_data"] = privacy_settings.get("anonymize_data", True)
                    self.privacy_settings["store_locally_only"] = privacy_settings.get("store_locally_only", True)
                    self.privacy_settings["allow_content_suggestions"] = privacy_settings.get("allow_content_suggestions", False)
            
            # Save settings to disk
            self._save_settings()
            
            # Run initial scan if enabled
            if self.enabled:
                self.scan_for_games()
        else:
            # User declined terms
            self.enabled = False
            self.terms_accepted = False
            
            # Reset any collected data
            self.detected_games = []
            
            # Save disabled state
            self._save_settings()
        
        print(f"Game detector {'enabled' if self.enabled else 'disabled'}")
        return self.enabled
    
    def _save_settings(self):
        """Save the current settings to disk"""
        settings_path = os.path.join("game", "data", "detector_settings.json")
        
        settings = {
            "enabled": self.enabled,
            "terms_accepted": self.terms_accepted,
            "terms_version": self.terms_version_accepted,
            "scan_settings": self.scan_settings,
            "privacy_settings": self.privacy_settings,
            "user_id": self.user_id,
            "last_scan_id": self.scan_id
        }
        
        try:
            with open(settings_path, "w") as f:
                json.dump(settings, f, indent=2)
            print(f"Settings saved to {settings_path}")
        except Exception as e:
            print(f"Error saving settings: {e}")
    
    def scan_for_games(self):
        """
        Scan the system for installed games
        Returns a list of detected games
        """
        if not self.enabled or not self.terms_accepted:
            return {"error": "Terms and conditions not accepted", "detected_games": []}
        
        self.detected_games = []
        start_time = time.time()
        
        # Scan default paths based on settings
        for path in self.game_paths:
            if os.path.exists(path):
                self._scan_directory(path, depth=0)
        
        # Scan additional game launchers if enabled
        if self.scan_settings.get("scan_steam", True):
            self._scan_steam_games()
        
        if self.scan_settings.get("scan_epic", True):
            self._scan_epic_games()
        
        # Additional launchers would be implemented here
        
        # Prepare result
        result = {
            "scan_id": self.scan_id,
            "user_id": self.user_id,
            "timestamp": time.time(),
            "scan_duration": time.time() - start_time,
            "detected_games": self.detected_games,
            "system_specs": self.system_specs if self.privacy_settings.get("collect_system_specs", True) else None
        }
        
        # Save scan results locally
        self._save_scan_results(result)
        
        return result
    
    def _scan_directory(self, path, depth=0):
        """Recursively scan a directory for games"""
        if depth > self.scan_settings.get("max_scan_depth", 2):
            return
            
        try:
            for item in os.listdir(path):
                item_path = os.path.join(path, item)
                
                # Check if this is a directory
                if os.path.isdir(item_path):
                    # Check if the folder name matches any game signatures
                    for game_name, signature in self.game_signatures.items():
                        folder_patterns = signature.get("folder_patterns", [])
                        for pattern in folder_patterns:
                            if re.search(pattern, item, re.IGNORECASE):
                                self._add_detected_game(game_name, item_path, "folder_match")
                                break
                    
                    # Recursively scan subdirectories
                    self._scan_directory(item_path, depth + 1)
                
                # Check executables
                elif item.endswith(".exe") and platform.system() == "Windows":
                    for game_name, signature in self.game_signatures.items():
                        exe_patterns = signature.get("exe_patterns", [])
                        for pattern in exe_patterns:
                            if re.search(pattern, item, re.IGNORECASE):
                                self._add_detected_game(game_name, item_path, "exe_match")
                                break
                
                # Check app bundles on macOS
                elif item.endswith(".app") and platform.system() == "Darwin":
                    for game_name, signature in self.game_signatures.items():
                        exe_patterns = signature.get("exe_patterns", [])
                        for pattern in exe_patterns:
                            # Remove .exe for macOS comparison
                            pattern = pattern.replace(".exe", "")
                            if re.search(pattern, item, re.IGNORECASE):
                                self._add_detected_game(game_name, item_path, "app_match")
                                break
        except Exception as e:
            # Just skip directories we can't access
            pass
    
    def _scan_steam_games(self):
        """Scan for Steam games"""
        # This would use Steam API or parse libraryfolders.vdf
        # For privacy reasons in this demo, we'll just return placeholders
        if platform.system() == "Windows":
            self._add_detected_game("Steam Game Example", "C:\\Program Files (x86)\\Steam\\steamapps\\common\\SteamGameExample", "steam")
        elif platform.system() == "Darwin":
            self._add_detected_game("Steam Game Example", "~/Library/Application Support/Steam/steamapps/common/SteamGameExample", "steam")
        elif platform.system() == "Linux":
            self._add_detected_game("Steam Game Example", "~/.steam/steam/steamapps/common/SteamGameExample", "steam")
    
    def _scan_epic_games(self):
        """Scan for Epic Games"""
        # This would parse the Epic Games manifest files
        # For privacy reasons in this demo, we'll just return placeholders
        if platform.system() == "Windows":
            self._add_detected_game("Epic Game Example", "C:\\Program Files\\Epic Games\\EpicGameExample", "epic")
        elif platform.system() == "Darwin":
            self._add_detected_game("Epic Game Example", "~/Library/Application Support/Epic/EpicGamesLauncher/EpicGameExample", "epic")
    
    def _add_detected_game(self, name, path, detection_method):
        """Add a game to the detected games list"""
        # Check if we already detected this game
        for game in self.detected_games:
            if game["name"] == name:
                # Already detected this game, add alternative path
                if path not in game["paths"]:
                    game["paths"].append(path)
                    game["detection_methods"].append(detection_method)
                return
        
        # Get metadata if available
        metadata = self.game_signatures.get(name, {})
        
        # Create new entry
        game_entry = {
            "name": name,
            "paths": [path],
            "detection_methods": [detection_method],
            "detected_at": time.time(),
            "metadata": {
                "category": metadata.get("category", "Unknown"),
                "genre": metadata.get("genre", ["Unknown"]),
                "developer": metadata.get("developer", "Unknown"),
                "publisher": metadata.get("publisher", "Unknown")
            }
        }
        
        self.detected_games.append(game_entry)
    
    def _save_scan_results(self, results):
        """Save scan results to file"""
        if not self.privacy_settings.get("store_locally_only", True):
            return  # Don't save if user opted out
            
        scan_dir = os.path.join("game", "data", "game_scans")
        os.makedirs(scan_dir, exist_ok=True)
        
        # Create filename based on date and scan ID
        date_str = time.strftime("%Y%m%d")
        scan_filename = f"game_scan_{date_str}_{self.scan_id[:8]}.json"
        
        # Save scan data
        scan_path = os.path.join(scan_dir, scan_filename)
        try:
            with open(scan_path, 'w') as f:
                # Anonymize if requested
                if self.privacy_settings.get("anonymize_data", True):
                    results_copy = dict(results)
                    if "user_id" in results_copy:
                        results_copy["user_id"] = hashlib.sha256(results_copy["user_id"].encode()).hexdigest()[:16]
                    json.dump(results_copy, f, indent=2)
                else:
                    json.dump(results, f, indent=2)
        except Exception as e:
            print(f"Error saving scan results: {e}")
    
    def get_game_recommendations(self):
        """Generate game recommendations based on detected games"""
        if not self.enabled or not self.terms_accepted:
            return {"error": "Terms and conditions not accepted", "recommendations": []}
            
        if not self.privacy_settings.get("allow_content_suggestions", True):
            return {"error": "Game recommendations disabled in privacy settings", "recommendations": []}
            
        if not self.detected_games:
            return {"error": "No games detected yet", "recommendations": []}
        
        # In a real implementation, this would use a recommendation algorithm
        # For this example, we'll just return some static recommendations
        recommendations = [
            {
                "name": "Warhammer 40,000: Dawn of War III",
                "reason": "Based on your interest in Warhammer 40K games",
                "confidence": 0.95,
                "metadata": {
                    "category": "RTS",
                    "genre": ["Strategy", "Warhammer", "Sci-Fi"],
                    "developer": "Relic Entertainment",
                    "publisher": "SEGA"
                }
            },
            {
                "name": "Total War: Warhammer III",
                "reason": "Based on your interest in Total War: Warhammer",
                "confidence": 0.9,
                "metadata": {
                    "category": "Strategy",
                    "genre": ["Turn-Based Strategy", "Real-Time Tactics", "Fantasy"],
                    "developer": "Creative Assembly",
                    "publisher": "SEGA"
                }
            },
            {
                "name": "Warhammer 40,000: Battlesector",
                "reason": "Based on your interest in turn-based strategy games",
                "confidence": 0.8,
                "metadata": {
                    "category": "Strategy",
                    "genre": ["Turn-Based Strategy", "Warhammer", "Sci-Fi"],
                    "developer": "Black Lab Games",
                    "publisher": "Slitherine Ltd."
                }
            }
        ]
        
        return {
            "user_id": self.user_id,
            "timestamp": time.time(),
            "based_on": [game["name"] for game in self.detected_games],
            "recommendations": recommendations
        }
    
    def get_terms_and_conditions(self):
        """Return the terms and conditions as a structured object for UI display"""
        return {
            "version": TERMS_VERSION,
            "terms": [
                {
                    "id": "intro",
                    "section": "Introduction",
                    "items": [
                        "By enabling the Game Detection feature, you agree to the following terms and conditions.",
                        f"Terms and Conditions Version: {TERMS_VERSION}",
                        "Last Updated: May 15, 2023",
                        "These terms govern the collection and use of data related to your gaming habits.",
                        "Please read carefully before accepting."
                    ]
                },
                {
                    "id": "data_collection",
                    "section": "Data Collection",
                    "items": [
                        "☐ I understand that this feature will scan my computer for installed games.",
                        "☐ I understand that scan results may include game names, installation paths, and metadata.",
                        "☐ I understand that scans may include data from Steam, Epic Games, and other gaming platforms.",
                        "☐ I understand that basic system specifications may be collected to improve recommendations.",
                        "☐ I understand that anonymous usage statistics may be collected to improve the feature.",
                        "☐ I understand that game playtime information may be collected if available.",
                        "☐ I understand that no personal files or documents will be accessed.",
                        "☐ I understand that scans will only occur when explicitly initiated or according to my preferences.",
                        "☐ I understand that I can disable specific scan targets in the settings.",
                        "☐ I understand that I can revoke this permission at any time.",
                        "☐ I understand that scan frequency can be configured in the settings."
                    ]
                },
                {
                    "id": "data_use",
                    "section": "Data Use",
                    "items": [
                        "☐ I understand that collected data will be used to enhance my gaming experience.",
                        "☐ I understand that my data will be used to provide game recommendations.",
                        "☐ I understand that my data may be used to suggest relevant game features.",
                        "☐ I understand that my data may be used to optimize performance on my hardware.",
                        "☐ I understand that aggregated, anonymized data may be used for development purposes.",
                        "☐ I understand that my data will not be sold to third parties.",
                        "☐ I understand that my data will not be used for targeted advertising.",
                        "☐ I understand that my individual data will not be shared without explicit consent.",
                        "☐ I understand that I can request a copy of my collected data at any time.",
                        "☐ I understand that I can request deletion of my collected data at any time."
                    ]
                },
                {
                    "id": "data_storage",
                    "section": "Data Storage",
                    "items": [
                        "☐ I understand that scan results are stored on my local device by default.",
                        "☐ I understand that I can choose to share anonymous data with the developers.",
                        "☐ I understand that data retention policies are configurable in settings.",
                        "☐ I understand that local data will be encrypted if supported by my system.",
                        "☐ I understand that data backups may be created during updates.",
                        "☐ I understand that shared data will be stored securely on developer servers if enabled.",
                        "☐ I understand that data will be automatically deleted after the configured retention period.",
                        "☐ I understand that I can manually delete all stored data at any time.",
                        "☐ I understand that data deletion is permanent and cannot be reversed.",
                        "☐ I understand that some aggregate statistical data may be retained even after deletion."
                    ]
                },
                {
                    "id": "privacy",
                    "section": "Privacy Protection",
                    "items": [
                        "☐ I understand that all data collection is opt-in and requires my explicit consent.",
                        "☐ I understand that my user ID is anonymized by default.",
                        "☐ I understand that I can choose to further anonymize collected data.",
                        "☐ I understand that I can review all collected data before sharing.",
                        "☐ I understand that I can exclude specific games from detection.",
                        "☐ I understand that I can configure scan depth and scope.",
                        "☐ I understand that scan results are private by default.",
                        "☐ I understand that the feature respects system-level 'Do Not Track' settings if enabled.",
                        "☐ I understand that personal identifiers are removed before data sharing.",
                        "☐ I understand that sensitive system information is not collected.",
                        "☐ I understand that scan logs are temporary and can be deleted."
                    ]
                },
                {
                    "id": "user_rights",
                    "section": "User Rights",
                    "items": [
                        "☐ I understand that I have the right to access all collected data.",
                        "☐ I understand that I have the right to correct inaccurate data.",
                        "☐ I understand that I have the right to delete all collected data.",
                        "☐ I understand that I have the right to withdraw consent at any time.",
                        "☐ I understand that I have the right to export my data in a portable format.",
                        "☐ I understand that I have the right to limit data collection scope.",
                        "☐ I understand that I have the right to object to data processing.",
                        "☐ I understand that I have the right to be informed about data practices.",
                        "☐ I understand that I have the right to restrict automated decision-making.",
                        "☐ I understand that I have the right to lodge a complaint with relevant authorities."
                    ]
                },
                {
                    "id": "benefits",
                    "section": "Feature Benefits",
                    "items": [
                        "☐ I understand that this feature enables personalized game recommendations.",
                        "☐ I understand that this feature may provide performance optimizations for my hardware.",
                        "☐ I understand that this feature may suggest appropriate difficulty settings.",
                        "☐ I understand that this feature may offer theme customization based on my preferences.",
                        "☐ I understand that this feature may suggest in-game content I might enjoy.",
                        "☐ I understand that this feature may help discover similar factions based on my interests.",
                        "☐ I understand that this feature may tailor game updates to my preferences.",
                        "☐ I understand that this feature may improve game compatibility with my system.",
                        "☐ I understand that this feature may suggest gameplay styles based on my habits.",
                        "☐ I understand that this feature may connect me with players of similar interests."
                    ]
                },
                {
                    "id": "limitations",
                    "section": "Limitations",
                    "items": [
                        "☐ I understand that game detection may not identify all installed games.",
                        "☐ I understand that recommendations are suggestions and not guarantees.",
                        "☐ I understand that system access limitations may affect detection accuracy.",
                        "☐ I understand that detection may have minimal performance impact during scans.",
                        "☐ I understand that detection requires adequate disk space for storing results.",
                        "☐ I understand that some features may not be available offline.",
                        "☐ I understand that detection accuracy may vary by operating system.",
                        "☐ I understand that third-party launchers may affect detection capabilities.",
                        "☐ I understand that frequently changing game installations may require rescans.",
                        "☐ I understand that system modifications may interfere with detection."
                    ]
                },
                {
                    "id": "updates",
                    "section": "Updates and Changes",
                    "items": [
                        "☐ I understand that these terms may be updated periodically.",
                        "☐ I understand that I will be notified of significant changes to these terms.",
                        "☐ I understand that continued use after updates constitutes acceptance.",
                        "☐ I understand that I may need to reaccept terms after major updates.",
                        "☐ I understand that new scanning capabilities may be added in updates.",
                        "☐ I understand that data usage policies may evolve over time.",
                        "☐ I understand that privacy protections will be maintained or enhanced in updates.",
                        "☐ I understand that I can withdraw consent after any terms update.",
                        "☐ I understand that legacy data may be subject to new terms after updates.",
                        "☐ I understand that I should review terms after each major software update."
                    ]
                },
                {
                    "id": "contact",
                    "section": "Contact and Support",
                    "items": [
                        "☐ I understand that I can contact support for questions about data collection.",
                        "☐ I understand that I can request data deletion through support channels.",
                        "☐ I understand that I can report privacy concerns to the developers.",
                        "☐ I understand that support can assist with feature configuration.",
                        "☐ I understand that feature documentation is available in the help section.",
                        "☐ I understand that privacy policy details are available on the website.",
                        "☐ I understand that FAQs about data collection are available in-game.",
                        "☐ I understand that the full privacy policy complements these terms.",
                        "☐ I understand that regulatory compliance information is available upon request.",
                        "☐ I understand that data breach notifications will be provided if required."
                    ]
                },
                {
                    "id": "acknowledgment",
                    "section": "Final Acknowledgment",
                    "items": [
                        "☐ I have read and understand all terms and conditions.",
                        "☐ I freely and voluntarily agree to these terms.",
                        "☐ I am of legal age to consent to data collection.",
                        "☐ I confirm that I have authority to consent on this device.",
                        "☐ I acknowledge that checking all boxes indicates full acceptance."
                    ]
                }
            ]
        }
    
    def get_status_report(self):
        """Generate a status report for the feature"""
        return {
            "enabled": self.enabled,
            "terms_accepted": self.terms_accepted,
            "terms_version": self.terms_version_accepted,
            "acceptance_date": self.terms_acceptance_date,
            "user_id": self.user_id,
            "scan_settings": self.scan_settings,
            "privacy_settings": self.privacy_settings,
            "detected_games_count": len(self.detected_games),
            "system_platform": platform.system(),
            "storage_path": os.path.join("game", "data", "game_scans")
        } 