"""
Faction-specific AI behaviors for units in the game
"""
import random

class FactionAI:
    """Base class for faction-specific AI behaviors"""
    def __init__(self, battle_scene):
        self.battle_scene = battle_scene
        
    def process_units(self, units):
        """Process all units of this faction - should be overridden by subclasses"""
        pass
    
    def _calculate_distance(self, pos1, pos2):
        """Calculate Manhattan distance between two grid positions"""
        x1, y1 = pos1
        x2, y2 = pos2
        return abs(x1 - x2) + abs(y1 - y2)
        
    def _try_move_unit(self, unit, new_x, new_y):
        """Try to move a unit to a new position, returns success/failure"""
        current_x, current_y = unit['position']
        grid = self.battle_scene.grid
        grid_size = self.battle_scene.grid_size
        
        # Check if the new position is valid and empty
        if not (0 <= new_x < grid_size[0] and 
                0 <= new_y < grid_size[1] and 
                grid[new_x][new_y] is None):
            return False
            
        # Get unit size for multi-tile units
        unit_size = unit.get('size', (1, 1))
        is_multi_tile = unit_size[0] > 1 or unit_size[1] > 1
        
        # Check if the entire unit can fit at the new position
        can_move = True
        if is_multi_tile:
            for udx in range(unit_size[0]):
                for udy in range(unit_size[1]):
                    check_x, check_y = new_x + udx, new_y + udy
                    if not (0 <= check_x < grid_size[0] and 
                            0 <= check_y < grid_size[1] and 
                            (grid[check_x][check_y] is None or grid[check_x][check_y] == unit)):
                        can_move = False
                        break
                if not can_move:
                    break
        
        if can_move:
            # Clear old position
            grid[current_x][current_y] = None
            
            # Clear all old positions for multi-tile units
            if is_multi_tile:
                for udx in range(unit_size[0]):
                    for udy in range(unit_size[1]):
                        old_x, old_y = current_x + udx, current_y + udy
                        if 0 <= old_x < grid_size[0] and 0 <= old_y < grid_size[1]:
                            if grid[old_x][old_y] == unit:
                                grid[old_x][old_y] = None
            
            # Set new position
            unit['position'] = (new_x, new_y)
            grid[new_x][new_y] = unit
            
            # Fill all new positions for multi-tile units
            if is_multi_tile:
                for udx in range(unit_size[0]):
                    for udy in range(unit_size[1]):
                        fill_x, fill_y = new_x + udx, new_y + udy
                        if 0 <= fill_x < grid_size[0] and 0 <= fill_y < grid_size[1]:
                            grid[fill_x][fill_y] = unit
            
            return True
        
        return False
        
    def _attack_unit(self, attacker, target):
        """Handle an AI unit attacking a player unit"""
        # This is a base implementation - override in subclasses for faction-specific behavior
        damage = max(5, attacker['attack'] - target['defense'] // 2) + random.randint(-3, 3)
        target['hp'] = max(0, target['hp'] - damage)
        
        self.battle_scene.battle_message = f"Enemy {attacker['name']} attacks {target['name']} for {damage} damage!"
        
        # Check if the player unit was defeated
        if target['hp'] <= 0:
            # Remove defeated player unit
            if target in self.battle_scene.player_units:
                self.battle_scene.player_units.remove(target)
            
            # Remove sprite
            if target["id"] in self.battle_scene.unit_sprites:
                del self.battle_scene.unit_sprites[target["id"]]
            
            # Clear unit from grid
            start_x, start_y = target['position']
            unit_size = target.get('size', (1, 1))
            
            # Clear all cells occupied by the unit
            for dx in range(unit_size[0]):
                for dy in range(unit_size[1]):
                    clear_x, clear_y = start_x + dx, start_y + dy
                    if 0 <= clear_x < self.battle_scene.grid_size[0] and 0 <= clear_y < self.battle_scene.grid_size[1]:
                        if self.battle_scene.grid[clear_x][clear_y] == target:
                            self.battle_scene.grid[clear_x][clear_y] = None
            
            self.battle_scene.battle_message = f"Enemy {attacker['name']} defeats {target['name']}!"
            
            # Check for game over
            if not self.battle_scene.player_units:
                self.battle_scene.battle_message = "Defeat! All your units are lost."
                # The battle scene handles the victory/defeat events
                return True
                
        return False

class OrkAI(FactionAI):
    """Ork AI behavior: simple, aggressive, and somewhat random"""
    
    def process_units(self, ork_units):
        """Process all Ork units according to their AI behavior"""
        for ork_unit in ork_units:
            # 20% chance an Ork will just do something random instead of logical
            if random.random() < 0.2:
                self._ork_random_action(ork_unit)
                continue
                
            # Find the closest player unit to attack - Orks always go for closest enemy
            nearest_player_unit = None
            nearest_distance = float('inf')
            
            for player_unit in self.battle_scene.player_units:
                distance = self._calculate_distance(ork_unit['position'], player_unit['position'])
                if distance < nearest_distance:
                    nearest_distance = distance
                    nearest_player_unit = player_unit
            
            if nearest_player_unit:
                # If in attack range, always attack
                if nearest_distance <= ork_unit['attack_range']:
                    self._attack_ork_style(ork_unit, nearest_player_unit)
                else:
                    # Move directly toward target, no tactical considerations
                    current_x, current_y = ork_unit['position']
                    target_x, target_y = nearest_player_unit['position']
                    
                    # Simple direct path - Orks don't think about optimal routes
                    dx = 1 if target_x > current_x else -1 if target_x < current_x else 0
                    dy = 1 if target_y > current_y else -1 if target_y < current_y else 0
                    
                    # Try to move in primary direction first
                    moved = False
                    if abs(target_x - current_x) > abs(target_y - current_y):
                        # Try horizontal movement first
                        if self._try_move_unit(ork_unit, current_x + dx, current_y):
                            moved = True
                        elif self._try_move_unit(ork_unit, current_x, current_y + dy):
                            moved = True
                    else:
                        # Try vertical movement first
                        if self._try_move_unit(ork_unit, current_x, current_y + dy):
                            moved = True
                        elif self._try_move_unit(ork_unit, current_x + dx, current_y):
                            moved = True
                            
                    if moved:
                        self.battle_scene.battle_message = f"Enemy {ork_unit['name']} charges toward {nearest_player_unit['name']}!"
    
    def _ork_random_action(self, ork_unit):
        """Orks sometimes do random things because they're not very smart"""
        # 50% chance to move in a random direction
        if random.random() < 0.5:
            current_x, current_y = ork_unit['position']
            
            # Pick a random direction
            directions = [(1, 0), (-1, 0), (0, 1), (0, -1)]
            random.shuffle(directions)
            
            # Try each direction until one works
            for dx, dy in directions:
                if self._try_move_unit(ork_unit, current_x + dx, current_y + dy):
                    self.battle_scene.battle_message = f"Enemy {ork_unit['name']} moves randomly. WAAAGH!"
                    break
        else:
            # Look for any player unit in attack range
            attackable_units = []
            for player_unit in self.battle_scene.player_units:
                distance = self._calculate_distance(ork_unit['position'], player_unit['position'])
                if distance <= ork_unit['attack_range']:
                    attackable_units.append(player_unit)
            
            # If can attack, attack a random target instead of nearest
            if attackable_units:
                target = random.choice(attackable_units)
                self._attack_ork_style(ork_unit, target)
    
    def _attack_ork_style(self, ork_unit, target):
        """Ork-specific attack with more randomness and WAAAGH! energy"""
        # Calculate and apply damage with extra randomness
        damage = max(5, ork_unit['attack'] - target['defense'] // 2) + random.randint(-5, 8)
        target['hp'] = max(0, target['hp'] - damage)
        
        # Ork-themed message
        self.battle_scene.battle_message = f"Enemy {ork_unit['name']} wildly attacks {target['name']} for {damage} damage! WAAAGH!"
        
        # Check if the player unit was defeated
        if target['hp'] <= 0:
            # Handle defeat (reusing parent class code)
            super()._attack_unit(ork_unit, target)
            # Override message with Ork flavor
            self.battle_scene.battle_message = f"Enemy {ork_unit['name']} crushes {target['name']}! WAAAGH!"

class TauAI(FactionAI):
    """Tau AI behavior: tactical, coordinated, and prioritizing ranged combat"""
    
    def process_units(self, tau_units):
        """Process all Tau units according to their AI behavior"""
        # First, analyze the battlefield and find optimal positions
        player_unit_positions = [unit['position'] for unit in self.battle_scene.player_units]
        
        # For each Tau unit, calculate a tactical plan
        for tau_unit in tau_units:
            # Assess if unit is in danger (enemy too close)
            in_danger = self._tau_assess_danger(tau_unit)
            
            # Get attack range - Tau prioritize using their superior ranged weapons
            attack_range = tau_unit['attack_range']
            
            if in_danger:
                # If in danger, retreat to a safe distance where they can still attack
                self._tau_tactical_retreat(tau_unit)
            else:
                # Find optimal firing position
                current_pos = tau_unit['position']
                
                # Find player units in range
                attackable_units = []
                for player_unit in self.battle_scene.player_units:
                    distance = self._calculate_distance(current_pos, player_unit['position'])
                    if distance <= attack_range:
                        attackable_units.append((player_unit, distance))
                
                if attackable_units:
                    # Tau are smart - they target the most damaged unit or highest-value target
                    target = self._tau_select_optimal_target(attackable_units)
                    self._attack_tau_style(tau_unit, target)
                else:
                    # Find optimal position to move to (cover as many enemies as possible with attack range)
                    optimal_pos = self._tau_find_optimal_position(tau_unit, player_unit_positions)
                    
                    if optimal_pos:
                        # Move toward optimal position
                        current_x, current_y = current_pos
                        target_x, target_y = optimal_pos
                        
                        # Calculate best path (more sophisticated than Ork AI)
                        path = self._calculate_path(current_pos, optimal_pos)
                        
                        if path and len(path) > 1:
                            next_pos = path[1]  # First step after current position
                            if self._try_move_unit(tau_unit, next_pos[0], next_pos[1]):
                                self.battle_scene.battle_message = f"Enemy {tau_unit['name']} takes a tactical position."
    
    def _tau_assess_danger(self, tau_unit):
        """Assess if a Tau unit is in immediate danger from player units"""
        tau_pos = tau_unit['position']
        
        # Tau consider themselves in danger if an enemy is too close
        for player_unit in self.battle_scene.player_units:
            distance = self._calculate_distance(tau_pos, player_unit['position'])
            
            # If a player unit is adjacent or very close, and has melee capability
            if distance <= 2:
                return True
                
        return False
        
    def _tau_tactical_retreat(self, tau_unit):
        """Move Tau unit to a safer position while maintaining firing capability"""
        current_pos = tau_unit['position']
        current_x, current_y = current_pos
        
        # Find the closest player unit (the threat)
        closest_threat = None
        closest_distance = float('inf')
        
        for player_unit in self.battle_scene.player_units:
            distance = self._calculate_distance(current_pos, player_unit['position'])
            if distance < closest_distance:
                closest_distance = distance
                closest_threat = player_unit
        
        if closest_threat:
            threat_x, threat_y = closest_threat['position']
            
            # Move away from threat while staying in range if possible
            retreat_directions = []
            
            # Calculate opposite direction from threat
            dx = -1 if threat_x > current_x else 1 if threat_x < current_x else 0
            dy = -1 if threat_y > current_y else 1 if threat_y < current_y else 0
            
            # Prioritize moving directly away
            retreat_directions.append((dx, dy))
            
            # Add orthogonal directions as alternatives
            if dx != 0:
                retreat_directions.append((dx, 0))
            if dy != 0:
                retreat_directions.append((0, dy))
                
            # Add diagonal retreat options
            retreat_directions.extend([(-1, -1), (-1, 1), (1, -1), (1, 1)])
            
            # Try each retreat direction
            for dx, dy in retreat_directions:
                new_x, new_y = current_x + dx, current_y + dy
                
                # Check if this position is far enough from threats
                is_safe = True
                for player_unit in self.battle_scene.player_units:
                    if self._calculate_distance((new_x, new_y), player_unit['position']) < 3:
                        is_safe = False
                        break
                
                if is_safe and self._try_move_unit(tau_unit, new_x, new_y):
                    self.battle_scene.battle_message = f"Enemy {tau_unit['name']} tactically retreats."
                    return
            
            # If no good retreat found, just try to move away
            for dx, dy in retreat_directions:
                if self._try_move_unit(tau_unit, current_x + dx, current_y + dy):
                    self.battle_scene.battle_message = f"Enemy {tau_unit['name']} falls back."
                    return
                    
    def _tau_select_optimal_target(self, attackable_units):
        """Tau select the optimal target based on damage potential and unit value"""
        highest_value_target = None
        highest_score = -1
        
        for unit, distance in attackable_units:
            # Score based on multiple factors (higher is better target)
            score = 0
            
            # Factor 1: Low HP targets are more valuable (finishing off units)
            hp_percent = unit['hp'] / unit['max_hp']
            score += (1 - hp_percent) * 10  # Up to 10 points for nearly dead units
            
            # Factor 2: High-value targets (elite units)
            if "elite" in unit.get("unit_type", ""):
                score += 5
            elif "heavy" in unit.get("unit_type", ""):
                score += 3
                
            # Factor 3: Closer targets are slightly preferred (more reliable hits)
            score += (5 - min(distance, 5))
            
            if score > highest_score:
                highest_score = score
                highest_value_target = unit
                
        # If no high-value target found, return the first one
        return highest_value_target or attackable_units[0][0]
    
    def _tau_find_optimal_position(self, tau_unit, player_positions):
        """Find an optimal position for a Tau unit to attack from"""
        current_pos = tau_unit['position']
        current_x, current_y = current_pos
        attack_range = tau_unit['attack_range']
        move_range = tau_unit.get('move_range', 3)
        
        # Generate potential positions within move range
        potential_positions = []
        for dx in range(-move_range, move_range + 1):
            for dy in range(-move_range, move_range + 1):
                if abs(dx) + abs(dy) <= move_range:
                    new_x, new_y = current_x + dx, current_y + dy
                    
                    # Check if position is valid
                    if (0 <= new_x < self.battle_scene.grid_size[0] and 
                        0 <= new_y < self.battle_scene.grid_size[1] and 
                        self.battle_scene.grid[new_x][new_y] is None):
                        
                        # Evaluate how many enemies would be in range from this position
                        targets_in_range = 0
                        for player_pos in player_positions:
                            if self._calculate_distance((new_x, new_y), player_pos) <= attack_range:
                                targets_in_range += 1
                                
                        # Score this position (consider both targets and safety)
                        position_score = targets_in_range * 10
                        
                        # Prefer positions farther from enemies (safer)
                        min_distance_to_enemy = float('inf')
                        for player_pos in player_positions:
                            min_distance_to_enemy = min(
                                min_distance_to_enemy, 
                                self._calculate_distance((new_x, new_y), player_pos)
                            )
                        
                        # Add distance bonus (up to 5 points)
                        position_score += min(min_distance_to_enemy, 5)
                        
                        potential_positions.append(((new_x, new_y), position_score))
        
        # Sort by score (descending) and return the best position
        potential_positions.sort(key=lambda x: x[1], reverse=True)
        
        return potential_positions[0][0] if potential_positions else None
    
    def _calculate_path(self, start_pos, end_pos):
        """Calculate a path from start to end (simple A* implementation)"""
        # Implementation of A* would go here
        # For now, just return direct path as a simplification
        path = [start_pos]
        
        # Just try to get closer in each step (Manhattan distance)
        current_x, current_y = start_pos
        end_x, end_y = end_pos
        
        # Calculate steps to take (simplistic path)
        x_steps = end_x - current_x
        y_steps = end_y - current_y
        
        # Create a path that first moves horizontally, then vertically
        # This is a simplification but works for most cases
        for _ in range(abs(x_steps)):
            current_x += 1 if x_steps > 0 else -1
            path.append((current_x, current_y))
            
        for _ in range(abs(y_steps)):
            current_y += 1 if y_steps > 0 else -1
            path.append((current_x, current_y))
            
        return path
        
    def _attack_tau_style(self, tau_unit, target):
        """Tau-specific attack with better accuracy"""
        # Calculate and apply damage with good accuracy (less randomness)
        damage = max(6, tau_unit['attack'] - target['defense'] // 3) + random.randint(-1, 3)
        target['hp'] = max(0, target['hp'] - damage)
        
        # Tau-themed message
        self.battle_scene.battle_message = f"Enemy {tau_unit['name']} precisely targets {target['name']} for {damage} damage."
        
        # Check if the player unit was defeated
        if target['hp'] <= 0:
            # Handle defeat (reusing parent class code)
            return super()._attack_unit(tau_unit, target)
        return False

class EldarAI(FactionAI):
    """Eldar AI behavior: lightning-fast strikes, psychic abilities, and tactical repositioning"""
    
    def process_units(self, eldar_units):
        """Process all Eldar units according to their AI behavior"""
        # First, analyze the battlefield and find best targets
        player_unit_positions = [unit['position'] for unit in self.battle_scene.player_units]
        
        for eldar_unit in eldar_units:
            # Determine unit role based on type
            unit_type = eldar_unit.get("unit_type", "")
            unit_name = eldar_unit.get("name", "").lower()
            
            # Special handling for different unit types
            if "farseer" in unit_name or "warlock" in unit_name or "elite" in unit_type:
                # Psychic units - prioritize special abilities and support
                self._handle_eldar_psyker(eldar_unit)
            elif "melee" in unit_type or "scorpion" in unit_name or "banshee" in unit_name:
                # Melee specialists - swift strikes
                self._handle_eldar_melee(eldar_unit)
            else:
                # Standard ranged units - repositioning and fire
                self._handle_eldar_ranged(eldar_unit)
    
    def _handle_eldar_psyker(self, eldar_unit):
        """Handle behavior for Eldar psykers (Farseers, Warlocks)"""
        current_pos = eldar_unit['position']
        attack_range = eldar_unit['attack_range']
        
        # Check if any player units are in attack range
        attackable_units = []
        for player_unit in self.battle_scene.player_units:
            distance = self._calculate_distance(current_pos, player_unit['position'])
            if distance <= attack_range:
                attackable_units.append(player_unit)
        
        if attackable_units:
            # Select the highest value target (elite units or damaged units)
            target = self._select_best_psychic_target(attackable_units)
            self._attack_eldar_psychic(eldar_unit, target)
        else:
            # Move to a position where they can attack next turn
            self._eldar_tactical_movement(eldar_unit, max_distance=3, min_distance=2)
    
    def _handle_eldar_melee(self, eldar_unit):
        """Handle behavior for Eldar melee specialists"""
        current_pos = eldar_unit['position']
        attack_range = eldar_unit['attack_range']
        
        # Check if any player units are in attack range
        attackable_units = []
        for player_unit in self.battle_scene.player_units:
            distance = self._calculate_distance(current_pos, player_unit['position'])
            if distance <= attack_range:
                attackable_units.append(player_unit)
        
        if attackable_units:
            # Select the weakest target for melee strike
            target = min(attackable_units, key=lambda unit: unit['hp'])
            self._attack_eldar_melee(eldar_unit, target)
        else:
            # Move aggressively toward nearest player unit
            nearest_player_unit = None
            nearest_distance = float('inf')
            
            for player_unit in self.battle_scene.player_units:
                distance = self._calculate_distance(current_pos, player_unit['position'])
                if distance < nearest_distance:
                    nearest_distance = distance
                    nearest_player_unit = player_unit
            
            if nearest_player_unit:
                self._eldar_swift_advance(eldar_unit, nearest_player_unit['position'])
    
    def _handle_eldar_ranged(self, eldar_unit):
        """Handle behavior for Eldar ranged units"""
        current_pos = eldar_unit['position']
        attack_range = eldar_unit['attack_range']
        
        # Check if any player units are in attack range
        attackable_units = []
        for player_unit in self.battle_scene.player_units:
            distance = self._calculate_distance(current_pos, player_unit['position'])
            if distance <= attack_range:
                attackable_units.append(player_unit)
        
        if attackable_units:
            # Select the best target for ranged attack
            target = self._select_best_ranged_target(attackable_units)
            self._attack_eldar_ranged(eldar_unit, target)
        else:
            # Reposition to firing position
            self._eldar_tactical_movement(eldar_unit, max_distance=4, min_distance=3)
    
    def _select_best_psychic_target(self, targets):
        """Select the best target for a psychic attack"""
        highest_value = -1
        best_target = None
        
        for target in targets:
            # Calculate target value
            value = 0
            
            # Elite units are high value
            if "elite" in target.get("unit_type", ""):
                value += 5
            elif "heavy" in target.get("unit_type", ""):
                value += 3
            
            # Higher level units are higher value
            value += target.get("level", 1) * 2
            
            # Damaged units are higher value (to finish them off)
            hp_ratio = target["hp"] / target["max_hp"]
            if hp_ratio < 0.5:
                value += 4
            
            if value > highest_value:
                highest_value = value
                best_target = target
        
        return best_target or targets[0]
    
    def _select_best_ranged_target(self, targets):
        """Select the best target for a ranged attack"""
        highest_value = -1
        best_target = None
        
        for target in targets:
            # Calculate target value
            value = 0
            
            # Prioritize units with high attack but low defense
            value += target.get("attack", 0) // 2
            value -= target.get("defense", 0) // 3
            
            # Damaged units are higher value (to finish them off)
            hp_ratio = target["hp"] / target["max_hp"]
            if hp_ratio < 0.3:
                value += 6
            elif hp_ratio < 0.5:
                value += 3
            
            if value > highest_value:
                highest_value = value
                best_target = target
        
        return best_target or targets[0]
    
    def _eldar_swift_advance(self, eldar_unit, target_pos):
        """Move an Eldar unit swiftly toward a target position"""
        current_pos = eldar_unit['position']
        current_x, current_y = current_pos
        target_x, target_y = target_pos
        
        # Calculate direction to target
        dx = 1 if target_x > current_x else -1 if target_x < current_x else 0
        dy = 1 if target_y > current_y else -1 if target_y < current_y else 0
        
        # Try diagonal movement first (Eldar are agile)
        if dx != 0 and dy != 0 and self._try_move_unit(eldar_unit, current_x + dx, current_y + dy):
            self.battle_scene.battle_message = f"Enemy {eldar_unit['name']} advances swiftly."
            return True
            
        # Try primary direction
        if abs(target_x - current_x) > abs(target_y - current_y):
            # Horizontal movement is more important
            if self._try_move_unit(eldar_unit, current_x + dx, current_y):
                self.battle_scene.battle_message = f"Enemy {eldar_unit['name']} advances toward target."
                return True
            elif self._try_move_unit(eldar_unit, current_x, current_y + dy):
                self.battle_scene.battle_message = f"Enemy {eldar_unit['name']} advances toward target."
                return True
        else:
            # Vertical movement is more important
            if self._try_move_unit(eldar_unit, current_x, current_y + dy):
                self.battle_scene.battle_message = f"Enemy {eldar_unit['name']} advances toward target."
                return True
            elif self._try_move_unit(eldar_unit, current_x + dx, current_y):
                self.battle_scene.battle_message = f"Enemy {eldar_unit['name']} advances toward target."
                return True
        
        # Try any other viable direction
        for move_dx, move_dy in [(0, 1), (1, 0), (0, -1), (-1, 0), (1, 1), (-1, -1), (1, -1), (-1, 1)]:
            if self._try_move_unit(eldar_unit, current_x + move_dx, current_y + move_dy):
                self.battle_scene.battle_message = f"Enemy {eldar_unit['name']} repositions."
                return True
                
        return False
    
    def _eldar_tactical_movement(self, eldar_unit, max_distance=3, min_distance=2):
        """Move an Eldar unit to maintain optimal distance from enemies"""
        current_pos = eldar_unit['position']
        current_x, current_y = current_pos
        
        # Find the best position that maintains distance from all enemies
        best_pos = None
        best_score = -999
        
        # Check all positions within move range
        move_range = eldar_unit.get('move_range', 4)
        for dx in range(-move_range, move_range + 1):
            for dy in range(-move_range, move_range + 1):
                if abs(dx) + abs(dy) <= move_range:
                    new_x, new_y = current_x + dx, current_y + dy
                    
                    # Check if position is valid
                    if (0 <= new_x < self.battle_scene.grid_size[0] and 
                        0 <= new_y < self.battle_scene.grid_size[1] and 
                        self.battle_scene.grid[new_x][new_y] is None):
                        
                        # Score this position
                        pos_score = 0
                        
                        # Check distance to all player units
                        for player_unit in self.battle_scene.player_units:
                            distance = self._calculate_distance((new_x, new_y), player_unit['position'])
                            
                            # Reward positions that are at optimal attack distance
                            if min_distance <= distance <= max_distance:
                                pos_score += 5
                            elif distance < min_distance:
                                pos_score -= 10  # Penalize getting too close
                            else:
                                pos_score -= (distance - max_distance)  # Slight penalty for being too far
                        
                        # Eldar prefer diagonal movement (more options)
                        if dx != 0 and dy != 0:
                            pos_score += 2
                        
                        if pos_score > best_score:
                            best_score = pos_score
                            best_pos = (new_x, new_y)
        
        # Move to the best position if found
        if best_pos:
            if self._try_move_unit(eldar_unit, best_pos[0], best_pos[1]):
                self.battle_scene.battle_message = f"Enemy {eldar_unit['name']} tactically repositions."
                return True
        
        # If no good position found, try random movement
        directions = [(0, 1), (1, 0), (0, -1), (-1, 0), (1, 1), (-1, -1), (1, -1), (-1, 1)]
        random.shuffle(directions)
        
        for dx, dy in directions:
            if self._try_move_unit(eldar_unit, current_x + dx, current_y + dy):
                self.battle_scene.battle_message = f"Enemy {eldar_unit['name']} shifts position."
                return True
                
        return False
    
    def _attack_eldar_melee(self, eldar_unit, target):
        """Eldar-specific melee attack with high damage but less randomness"""
        # Eldar melee specialists are deadly
        damage = max(7, eldar_unit['attack'] - target['defense'] // 3) + random.randint(0, 4)
        target['hp'] = max(0, target['hp'] - damage)
        
        # Eldar-themed message
        self.battle_scene.battle_message = f"Enemy {eldar_unit['name']} strikes {target['name']} with lightning speed for {damage} damage."
        
        # Check if the player unit was defeated
        if target['hp'] <= 0:
            # Handle defeat (reusing parent class code)
            return super()._attack_unit(eldar_unit, target)
        return False
    
    def _attack_eldar_ranged(self, eldar_unit, target):
        """Eldar-specific ranged attack with high accuracy"""
        # Eldar ranged attacks are precise
        damage = max(5, eldar_unit['attack'] - target['defense'] // 4) + random.randint(-1, 2)
        target['hp'] = max(0, target['hp'] - damage)
        
        # Eldar-themed message
        self.battle_scene.battle_message = f"Enemy {eldar_unit['name']} fires upon {target['name']} with deadly precision for {damage} damage."
        
        # Check if the player unit was defeated
        if target['hp'] <= 0:
            # Handle defeat (reusing parent class code)
            return super()._attack_unit(eldar_unit, target)
        return False
    
    def _attack_eldar_psychic(self, eldar_unit, target):
        """Eldar-specific psychic attack that bypasses some defense"""
        # Psychic attacks ignore some defense
        defense_factor = target['defense'] // 5  # Only 20% of defense applies
        damage = max(6, eldar_unit['attack'] - defense_factor) + random.randint(0, 3)
        target['hp'] = max(0, target['hp'] - damage)
        
        # Eldar-themed message
        self.battle_scene.battle_message = f"Enemy {eldar_unit['name']} unleashes psychic energy at {target['name']} for {damage} damage."
        
        # Check if the player unit was defeated
        if target['hp'] <= 0:
            # Handle defeat (reusing parent class code)
            return super()._attack_unit(eldar_unit, target)
        return False

class DefaultAI(FactionAI):
    """Default AI behavior for factions without specific implementation"""
    
    def process_units(self, units):
        """Process all units with generic AI behavior"""
        for enemy_unit in units:
            # Find nearest player unit
            nearest_player_unit = None
            nearest_distance = float('inf')
            
            for player_unit in self.battle_scene.player_units:
                distance = self._calculate_distance(enemy_unit['position'], player_unit['position'])
                if distance < nearest_distance:
                    nearest_distance = distance
                    nearest_player_unit = player_unit
            
            if nearest_player_unit:
                # If in attack range, attack
                if nearest_distance <= enemy_unit['attack_range']:
                    self._attack_unit(enemy_unit, nearest_player_unit)
                else:
                    # Move closer to target
                    current_x, current_y = enemy_unit['position']
                    target_x, target_y = nearest_player_unit['position']
                    
                    # Simple pathfinding: try to move in the direction of the target
                    dx = 0
                    dy = 0
                    
                    # Determine direction to move (prioritize x or y based on which is farther)
                    if abs(target_x - current_x) > abs(target_y - current_y):
                        dx = 1 if target_x > current_x else -1 if target_x < current_x else 0
                        dy = 1 if target_y > current_y else -1 if target_y < current_y else 0
                    else:
                        dy = 1 if target_y > current_y else -1 if target_y < current_y else 0
                        dx = 1 if target_x > current_x else -1 if target_x < current_x else 0
                    
                    # Try to move
                    if self._try_move_unit(enemy_unit, current_x + dx, current_y + dy) or \
                       self._try_move_unit(enemy_unit, current_x + dx, current_y) or \
                       self._try_move_unit(enemy_unit, current_x, current_y + dy):
                        self.battle_scene.battle_message = f"Enemy {enemy_unit['name']} moves toward {nearest_player_unit['name']}." 