"""
Analytics Module
Collects and processes game analytics data
"""
import os
import json
import time
import datetime
import uuid
from collections import defaultdict

class AnalyticsManager:
    """Manages collection, storage, and processing of game analytics data"""
    
    def __init__(self, game_manager=None):
        self.game_manager = game_manager
        self.enabled = True
        self.user_id = self._get_or_create_user_id()
        self.session_id = str(uuid.uuid4())
        self.session_start_time = time.time()
        
        # Data storage
        self.events = []
        self.session_data = {
            "session_id": self.session_id,
            "user_id": self.user_id,
            "start_time": self.session_start_time,
            "end_time": None,
            "game_version": game_manager.version if game_manager else "unknown",
            "platform": self._get_platform(),
            "settings": game_manager.settings if game_manager else {},
        }
        
        # Aggregated metrics
        self.metrics = defaultdict(int)
        self.timers = {}
        
        # Paths
        self.data_dir = os.path.join("game", "data", "analytics")
        self.ensure_data_dir()
        
        # Track this session
        self.track_event("session_start", {
            "screen_resolution": self._get_screen_resolution(),
            "fullscreen": game_manager.settings.get("fullscreen", False) if game_manager else False
        })
    
    def ensure_data_dir(self):
        """Ensure the analytics data directory exists"""
        os.makedirs(self.data_dir, exist_ok=True)
    
    def _get_or_create_user_id(self):
        """Get existing user ID or create a new one"""
        user_id_file = os.path.join("game", "data", "user_id.json")
        
        if os.path.exists(user_id_file):
            try:
                with open(user_id_file, 'r') as f:
                    data = json.load(f)
                    return data.get("user_id", str(uuid.uuid4()))
            except:
                pass
        
        # Create new user ID
        user_id = str(uuid.uuid4())
        os.makedirs(os.path.dirname(user_id_file), exist_ok=True)
        
        try:
            with open(user_id_file, 'w') as f:
                json.dump({"user_id": user_id, "created_at": time.time()}, f)
        except:
            pass
            
        return user_id
    
    def _get_platform(self):
        """Detect the platform"""
        import platform
        return {
            "system": platform.system(),
            "release": platform.release(),
            "version": platform.version(),
            "machine": platform.machine(),
            "processor": platform.processor()
        }
    
    def _get_screen_resolution(self):
        """Get the screen resolution if pygame is initialized"""
        if self.game_manager and hasattr(self.game_manager, "screen"):
            try:
                return {
                    "width": self.game_manager.screen.get_width(),
                    "height": self.game_manager.screen.get_height()
                }
            except:
                pass
        return {"width": 0, "height": 0}
    
    def track_event(self, event_type, properties=None):
        """
        Track a game event
        
        Args:
            event_type (str): Type of event (e.g., 'battle_start', 'mission_complete')
            properties (dict): Additional properties for the event
        """
        if not self.enabled:
            return
            
        event = {
            "event_id": str(uuid.uuid4()),
            "event_type": event_type,
            "timestamp": time.time(),
            "session_id": self.session_id,
            "user_id": self.user_id,
            "properties": properties or {}
        }
        
        # Add to events list
        self.events.append(event)
        
        # Update metrics
        self.metrics[f"event_{event_type}"] += 1
        
        # Auto-save if we have many events
        if len(self.events) >= 100:
            self.save_data()
    
    def track_battle(self, battle_data):
        """
        Track battle-specific data
        
        Args:
            battle_data (dict): Data about the battle
        """
        battle_id = str(uuid.uuid4())
        event_data = {
            "battle_id": battle_id,
            "mission_id": battle_data.get("mission_id", "unknown"),
            "enemy_faction": battle_data.get("enemy_faction", "unknown"),
            "difficulty": battle_data.get("difficulty", "Normal"),
            "player_units": battle_data.get("player_units", []),
            "enemy_units": battle_data.get("enemy_units", []),
            "result": battle_data.get("result", "unknown"),
            "turns": battle_data.get("turns", 0),
            "duration_seconds": battle_data.get("duration", 0),
            "player_units_lost": battle_data.get("player_units_lost", 0),
            "enemy_units_killed": battle_data.get("enemy_units_killed", 0)
        }
        
        self.track_event("battle_completed", event_data)
        
        # Update aggregated metrics
        self.metrics["battles_total"] += 1
        self.metrics[f"battles_against_{event_data['enemy_faction']}"] += 1
        
        if event_data["result"] == "victory":
            self.metrics["battles_won"] += 1
            self.metrics[f"battles_won_against_{event_data['enemy_faction']}"] += 1
        elif event_data["result"] == "defeat":
            self.metrics["battles_lost"] += 1
        
        self.metrics["total_turns"] += event_data["turns"]
        self.metrics["total_battle_time"] += event_data["duration_seconds"]
        self.metrics["total_player_units_lost"] += event_data["player_units_lost"]
        self.metrics["total_enemy_units_killed"] += event_data["enemy_units_killed"]
    
    def start_timer(self, timer_name):
        """Start a timer for tracking durations"""
        self.timers[timer_name] = time.time()
    
    def stop_timer(self, timer_name, event_name=None, properties=None):
        """
        Stop a timer and optionally track as an event
        
        Args:
            timer_name (str): Name of the timer to stop
            event_name (str, optional): If provided, create an event with the duration
            properties (dict, optional): Additional properties for the event
        
        Returns:
            float: Duration in seconds, or None if timer wasn't started
        """
        if timer_name not in self.timers:
            return None
            
        start_time = self.timers[timer_name]
        duration = time.time() - start_time
        
        # Update metrics
        self.metrics[f"timer_{timer_name}_count"] += 1
        self.metrics[f"timer_{timer_name}_total"] += duration
        
        # Remove the timer
        del self.timers[timer_name]
        
        # Create event if requested
        if event_name:
            props = properties or {}
            props["duration"] = duration
            self.track_event(event_name, props)
            
        return duration
    
    def save_data(self):
        """Save analytics data to disk"""
        if not self.enabled or not self.events:
            return
            
        # Update session data
        self.session_data["end_time"] = time.time()
        self.session_data["duration"] = self.session_data["end_time"] - self.session_data["start_time"]
        self.session_data["event_count"] = len(self.events)
        self.session_data["metrics"] = dict(self.metrics)
        
        # Create filenames based on date and session ID
        date_str = datetime.datetime.now().strftime("%Y%m%d")
        session_filename = f"session_{self.session_id}_{date_str}.json"
        events_filename = f"events_{self.session_id}_{date_str}.json"
        
        # Save session data
        session_path = os.path.join(self.data_dir, session_filename)
        try:
            with open(session_path, 'w') as f:
                json.dump(self.session_data, f)
        except Exception as e:
            print(f"Error saving session data: {e}")
        
        # Save events data
        events_path = os.path.join(self.data_dir, events_filename)
        try:
            with open(events_path, 'w') as f:
                json.dump(self.events, f)
            # Clear events after saving
            self.events = []
        except Exception as e:
            print(f"Error saving events data: {e}")
    
    def end_session(self):
        """End the current session and save data"""
        self.track_event("session_end", {
            "duration": time.time() - self.session_start_time
        })
        self.save_data()
    
    def get_metrics_report(self):
        """Generate a human-readable metrics report"""
        lines = [
            "=== GAME ANALYTICS REPORT ===",
            f"User ID: {self.user_id}",
            f"Session ID: {self.session_id}",
            f"Session Duration: {time.time() - self.session_start_time:.2f} seconds",
            "",
            "--- METRICS ---"
        ]
        
        # Add metrics
        for key, value in sorted(self.metrics.items()):
            lines.append(f"{key}: {value}")
        
        return "\n".join(lines)
    
    def export_data(self, format="json"):
        """
        Export all collected analytics data
        
        Args:
            format (str): Export format ('json' or 'csv')
            
        Returns:
            str: Path to the exported file
        """
        if format not in ["json", "csv"]:
            format = "json"
            
        filename = f"analytics_export_{datetime.datetime.now().strftime('%Y%m%d_%H%M%S')}.{format}"
        export_path = os.path.join(self.data_dir, filename)
        
        data = {
            "session": self.session_data,
            "events": self.events,
            "metrics": dict(self.metrics),
            "export_time": time.time()
        }
        
        try:
            if format == "json":
                with open(export_path, 'w') as f:
                    json.dump(data, f, indent=2)
            elif format == "csv":
                # Simplified CSV export
                import csv
                with open(export_path, 'w', newline='') as f:
                    writer = csv.writer(f)
                    writer.writerow(["event_type", "timestamp", "properties"])
                    for event in self.events:
                        writer.writerow([
                            event["event_type"],
                            event["timestamp"],
                            json.dumps(event["properties"])
                        ])
            
            return export_path
        except Exception as e:
            print(f"Error exporting data: {e}")
            return None 