import random

def run_tau_ai(battle_scene, tau_units):
    """Tau AI behavior: tactical, coordinated, and prioritizing ranged combat"""
    # First, analyze the battlefield to find optimal positions
    player_unit_positions = [unit['position'] for unit in battle_scene.player_units]
    
    # For each Tau unit, calculate a tactical plan
    for tau_unit in tau_units:
        # Assess if unit is in danger (enemy too close)
        in_danger = False
        tau_pos = tau_unit['position']
        
        # Check for player units that are too close (danger assessment)
        for player_unit in battle_scene.player_units:
            distance = battle_scene._calculate_distance(tau_pos, player_unit['position'])
            if distance <= 2:  # If a player unit is adjacent or very close
                in_danger = True
                break
                
        # Get attack range - Tau prioritize using their superior ranged weapons
        attack_range = tau_unit['attack_range']
        
        if in_danger:
            # If in danger, retreat to a safer position
            tactical_retreat(battle_scene, tau_unit)
        else:
            # Find player units in attack range
            attackable_units = []
            for player_unit in battle_scene.player_units:
                distance = battle_scene._calculate_distance(tau_pos, player_unit['position'])
                if distance <= attack_range:
                    attackable_units.append((player_unit, distance))
            
            if attackable_units:
                # Tau are smart - they target optimal units based on strategic value
                target = select_optimal_target(battle_scene, attackable_units)
                
                # Attack with Tau precision and themed message
                damage = max(5, tau_unit['attack'] - target['defense'] // 2) + random.randint(-2, 2)
                # Tau bonus damage for superior weapons
                damage = int(damage * 1.15)  # 15% bonus
                target['hp'] = max(0, target['hp'] - damage)
                battle_scene.battle_message = f"Enemy {tau_unit['name']} precisely targets {target['name']} for {damage} damage."
                
                # Check if the player unit was defeated
                if target['hp'] <= 0:
                    battle_scene._handle_player_unit_defeat(tau_unit, target)
                    battle_scene.battle_message = f"Enemy {tau_unit['name']} eliminates {target['name']} for the Greater Good."
            else:
                # Find optimal position to move to (cover as many enemies as possible)
                optimal_pos = find_optimal_position(battle_scene, tau_unit, player_unit_positions)
                
                if optimal_pos:
                    # Move toward optimal position
                    current_x, current_y = tau_pos
                    target_x, target_y = optimal_pos
                    
                    # Calculate best direction to move
                    dx = 1 if target_x > current_x else -1 if target_x < current_x else 0
                    dy = 1 if target_y > current_y else -1 if target_y < current_y else 0
                    
                    # Try to move with a more intelligent approach
                    if target_x == current_x:
                        # Only vertical movement needed
                        if battle_scene._try_move_enemy_unit(tau_unit, current_x, current_y + dy):
                            battle_scene.battle_message = f"Enemy {tau_unit['name']} takes a tactical position."
                    elif target_y == current_y:
                        # Only horizontal movement needed
                        if battle_scene._try_move_enemy_unit(tau_unit, current_x + dx, current_y):
                            battle_scene.battle_message = f"Enemy {tau_unit['name']} takes a tactical position."
                    else:
                        # Diagonal path needed - try horizontal first then vertical
                        moved = False
                        # First try x, then y
                        if battle_scene._try_move_enemy_unit(tau_unit, current_x + dx, current_y):
                            moved = True
                        elif battle_scene._try_move_enemy_unit(tau_unit, current_x, current_y + dy):
                            moved = True
                            
                        if moved:
                            battle_scene.battle_message = f"Enemy {tau_unit['name']} repositions strategically."

def tactical_retreat(battle_scene, tau_unit):
    """Move Tau unit away from nearby player units to maintain ranged advantage"""
    current_x, current_y = tau_unit['position']
    
    # Check where player units are to find safe retreat direction
    retreat_scores = {'up': 0, 'down': 0, 'left': 0, 'right': 0}
    
    # Examine all player units
    for player_unit in battle_scene.player_units:
        px, py = player_unit['position']
        
        # Calculate directional scores - higher is better for retreat
        if py < current_y:  # Player is above, retreat down
            retreat_scores['down'] += 2
            retreat_scores['up'] -= 2
        elif py > current_y:  # Player is below, retreat up
            retreat_scores['up'] += 2
            retreat_scores['down'] -= 2
            
        if px < current_x:  # Player is left, retreat right
            retreat_scores['right'] += 2
            retreat_scores['left'] -= 2
        elif px > current_x:  # Player is right, retreat left
            retreat_scores['left'] += 2
            retreat_scores['right'] -= 2
    
    # Sort retreat directions by score (best first)
    directions = ['up', 'down', 'left', 'right']
    directions.sort(key=lambda dir: retreat_scores[dir], reverse=True)
    
    # Try each direction in order of preference
    for direction in directions:
        dx, dy = 0, 0
        if direction == 'up':
            dy = -1
        elif direction == 'down':
            dy = 1
        elif direction == 'left':
            dx = -1
        elif direction == 'right':
            dx = 1
            
        # Try the retreat move
        if battle_scene._try_move_enemy_unit(tau_unit, current_x + dx, current_y + dy):
            battle_scene.battle_message = f"Enemy {tau_unit['name']} tactically retreats."
            return True
    
    # If no retreat was possible, try any valid move to escape
    alternate_directions = [(0, 1), (1, 0), (0, -1), (-1, 0)]
    random.shuffle(alternate_directions)  # Try random directions
    
    for dx, dy in alternate_directions:
        if battle_scene._try_move_enemy_unit(tau_unit, current_x + dx, current_y + dy):
            battle_scene.battle_message = f"Enemy {tau_unit['name']} is forced to reposition."
            return True
    
    return False

def select_optimal_target(battle_scene, attackable_units):
    """Select the best target for a Tau unit to attack based on strategic value"""
    if not attackable_units:
        return None
    
    # Initial scoring based on simple metrics
    scored_targets = []
    
    for target_unit, distance in attackable_units:
        # Start with base score
        score = 0
        
        # Prefer low HP targets (easier to eliminate)
        hp_percentage = target_unit['hp'] / target_unit['max_hp']
        if hp_percentage < 0.3:
            score += 30  # Priority to finish off nearly-dead units
        elif hp_percentage < 0.5:
            score += 20
        
        # Target high-value units
        if target_unit.get('is_commander', False):
            score += 25  # Commanders are high value
        
        # Prefer closer targets slightly (more reliable hits)
        if distance <= 1:
            score += 5
        
        # Target offensive units first
        if target_unit.get('attack', 0) > 8:
            score += 15  # Prioritize taking out hard-hitting units
        
        # Target units that might threaten Tau positioning
        if target_unit.get('move_range', 0) > 4:
            score += 10  # Mobile units can flank - eliminate them
        
        scored_targets.append((target_unit, score))
    
    # Sort by score, highest first
    scored_targets.sort(key=lambda item: item[1], reverse=True)
    
    # Return the best target
    return scored_targets[0][0]

def find_optimal_position(battle_scene, tau_unit, player_positions):
    """Find the best position for a Tau unit to move to"""
    current_pos = tau_unit['position']
    attack_range = tau_unit['attack_range']
    
    # Get all possible move destinations
    move_range = battle_scene._calculate_move_range(current_pos, tau_unit['move_range'])
    if not move_range:
        return None  # No valid moves
    
    # Score each possible position
    scored_positions = []
    
    for pos in move_range:
        score = 0
        
        # Calculate how many player units could be attacked from this position
        attackable_count = 0
        for player_pos in player_positions:
            distance = battle_scene._calculate_distance(pos, player_pos)
            if distance <= attack_range:
                attackable_count += 1
                
                # Bonus for being exactly at max range (safest position to fire from)
                if distance == attack_range:
                    score += 5
        
        # Major bonus for positions that can attack multiple targets
        score += attackable_count * 20
        
        # Prefer positions that maintain distance from enemies (for safety)
        min_distance_to_enemy = float('inf')
        for player_pos in player_positions:
            distance = battle_scene._calculate_distance(pos, player_pos)
            min_distance_to_enemy = min(min_distance_to_enemy, distance)
        
        # Safety bonus - prefer positions further from enemies
        if min_distance_to_enemy > 2:
            score += 15
        elif min_distance_to_enemy > 1:
            score += 5
            
        # Add terrain bonus if available
        x, y = pos
        if 0 <= x < len(battle_scene.terrain_grid) and 0 <= y < len(battle_scene.terrain_grid[0]):
            terrain_type = battle_scene.terrain_grid[x][y]
            defense_bonus = battle_scene.terrain_types[terrain_type]["defense_bonus"]
            score += defense_bonus * 2
        
        scored_positions.append((pos, score))
    
    # Sort by score, highest first
    scored_positions.sort(key=lambda item: item[1], reverse=True)
    
    # Return the best position, or None if no good positions
    if scored_positions and scored_positions[0][1] > 0:
        return scored_positions[0][0]
        
    # If no good scoring position, return the position furthest from enemies
    max_distance_pos = None
    max_distance = 0
    
    for pos in move_range:
        min_distance = float('inf')
        for player_pos in player_positions:
            distance = battle_scene._calculate_distance(pos, player_pos)
            min_distance = min(min_distance, distance)
            
        if min_distance > max_distance:
            max_distance = min_distance
            max_distance_pos = pos
    
    return max_distance_pos 