import random

def run_ork_ai(battle_scene, ork_units):
    """Ork AI behavior: simple, aggressive, and somewhat random"""
    for ork_unit in ork_units:
        # 20% chance an Ork will just do something random instead of logical
        if random.random() < 0.2:
            # Random action: either move randomly or attack random target in range
            if random.random() < 0.5:
                # Move in a random direction
                current_x, current_y = ork_unit['position']
                # Pick a random direction
                directions = [(0, 1), (1, 0), (0, -1), (-1, 0)]
                random.shuffle(directions)
                # Try each direction until one works
                for dx, dy in directions:
                    new_x, new_y = current_x + dx, current_y + dy
                    # Try to move - reusing existing movement logic
                    if battle_scene._try_move_enemy_unit(ork_unit, new_x, new_y):
                        battle_scene.battle_message = f"Enemy {ork_unit['name']} moves randomly. WAAAGH!"
                        break
            else:
                # Look for any PLAYER unit in attack range (not other Orks)
                attackable_units = []
                for player_unit in battle_scene.player_units:  # Make sure we're only targeting player units
                    distance = battle_scene._calculate_distance(ork_unit['position'], player_unit['position'])
                    if distance <= ork_unit['attack_range']:
                        attackable_units.append(player_unit)
                
                # If can attack, attack a random player unit
                if attackable_units:
                    target = random.choice(attackable_units)
                    # Attack with higher randomness and Ork-themed message
                    damage = max(5, ork_unit['attack'] - target['defense'] // 2) + random.randint(-5, 8)
                    target['hp'] = max(0, target['hp'] - damage)
                    battle_scene.battle_message = f"Enemy {ork_unit['name']} wildly attacks {target['name']} for {damage} damage! WAAAGH!"
                    
                    # Check if the player unit was defeated
                    if target['hp'] <= 0:
                        battle_scene._handle_player_unit_defeat(ork_unit, target)
                        battle_scene.battle_message = f"Enemy {ork_unit['name']} crushes {target['name']}! WAAAGH!"
            continue
        
        # Standard behavior: Find nearest player unit
        nearest_player_unit = None
        nearest_distance = float('inf')
        
        for player_unit in battle_scene.player_units:
            distance = battle_scene._calculate_distance(ork_unit['position'], player_unit['position'])
            if distance < nearest_distance:
                nearest_distance = distance
                nearest_player_unit = player_unit
        
        if nearest_player_unit:
            # If in attack range, always attack
            if nearest_distance <= ork_unit['attack_range']:
                # Attack with higher randomness and Ork-themed message
                damage = max(5, ork_unit['attack'] - nearest_player_unit['defense'] // 2) + random.randint(-5, 8)
                nearest_player_unit['hp'] = max(0, nearest_player_unit['hp'] - damage)
                battle_scene.battle_message = f"Enemy {ork_unit['name']} wildly attacks {nearest_player_unit['name']} for {damage} damage! WAAAGH!"
                
                # Check if the player unit was defeated
                if nearest_player_unit['hp'] <= 0:
                    battle_scene._handle_player_unit_defeat(ork_unit, nearest_player_unit)
                    battle_scene.battle_message = f"Enemy {ork_unit['name']} crushes {nearest_player_unit['name']}! WAAAGH!"
            else:
                # Move directly toward target, no tactical considerations
                current_x, current_y = ork_unit['position']
                target_x, target_y = nearest_player_unit['position']
                
                # Simple direct path - Orks don't think about optimal routes
                dx = 1 if target_x > current_x else -1 if target_x < current_x else 0
                dy = 1 if target_y > current_y else -1 if target_y < current_y else 0
                
                # Try to move in primary direction first
                moved = False
                if abs(target_x - current_x) > abs(target_y - current_y):
                    # Try horizontal movement first
                    if battle_scene._try_move_enemy_unit(ork_unit, current_x + dx, current_y):
                        moved = True
                    elif battle_scene._try_move_enemy_unit(ork_unit, current_x, current_y + dy):
                        moved = True
                else:
                    # Try vertical movement first
                    if battle_scene._try_move_enemy_unit(ork_unit, current_x, current_y + dy):
                        moved = True
                    elif battle_scene._try_move_enemy_unit(ork_unit, current_x + dx, current_y):
                        moved = True
                        
                if moved:
                    battle_scene.battle_message = f"Enemy {ork_unit['name']} charges toward {nearest_player_unit['name']}!" 