import random

def run_chaos_ai(battle_scene, chaos_units):
    """Chaos AI behavior: unpredictable, aggressive, with occasional warp powers"""
    for chaos_unit in chaos_units:
        # Determine if this unit will use warp power this turn (25% chance)
        use_warp_power = random.random() < 0.25 and "warp_powers" in chaos_unit.get("abilities", [])
        
        # Determine if unit will enter berserker state (blood for the blood god!)
        # Higher chance if low health
        berserker_chance = 0.15
        if chaos_unit["hp"] < chaos_unit["max_hp"] * 0.4:
            berserker_chance = 0.40  # More likely when wounded
            
        enter_berserker = random.random() < berserker_chance
        
        # Find nearest player unit
        nearest_player_unit = None
        nearest_distance = float('inf')
        
        for player_unit in battle_scene.player_units:
            distance = battle_scene._calculate_distance(chaos_unit['position'], player_unit['position'])
            if distance < nearest_distance:
                nearest_distance = distance
                nearest_player_unit = player_unit
        
        if not nearest_player_unit:
            continue  # No player units to target
            
        # If using warp power and there's a target, prioritize that
        if use_warp_power:
            # Apply warp-based attack - can hit beyond normal attack range
            extended_range = chaos_unit.get('attack_range', 2) + 2
            if nearest_distance <= extended_range:
                # Warp power is unpredictable and can backfire
                if random.random() < 0.15:  # 15% chance of backfire
                    # Warp power backfires, causing damage to self
                    self_damage = random.randint(3, 8)
                    chaos_unit['hp'] = max(1, chaos_unit['hp'] - self_damage)  # Won't kill self
                    battle_scene.battle_message = f"Enemy {chaos_unit['name']}'s warp powers backfire, causing {self_damage} damage to itself!"
                else:
                    # Warp attack succeeds
                    warp_damage = max(10, chaos_unit['attack']) + random.randint(0, 10)
                    nearest_player_unit['hp'] = max(0, nearest_player_unit['hp'] - warp_damage)
                    battle_scene.battle_message = f"Enemy {chaos_unit['name']} unleashes chaotic warp energy at {nearest_player_unit['name']} for {warp_damage} damage!"
                    
                    # Check if the player unit was defeated
                    if nearest_player_unit['hp'] <= 0:
                        battle_scene._handle_player_unit_defeat(chaos_unit, nearest_player_unit)
                        battle_scene.battle_message = f"Enemy {chaos_unit['name']} obliterates {nearest_player_unit['name']} with warp energy! Blood for the Blood God!"
                continue
        
        # Berserker behavior - move aggressively, attack with enhanced strength
        if enter_berserker:
            # If in range, attack with enhanced damage
            if nearest_distance <= chaos_unit['attack_range']:
                # Berserker attacks have higher damage but less precision
                berserker_damage = int(chaos_unit['attack'] * 1.4) + random.randint(-3, 10)
                nearest_player_unit['hp'] = max(0, nearest_player_unit['hp'] - berserker_damage)
                battle_scene.battle_message = f"Enemy {chaos_unit['name']} enters a blood frenzy, savagely attacking {nearest_player_unit['name']} for {berserker_damage} damage!"
                
                # Berserker can harm itself in frenzy
                if random.random() < 0.2:  # 20% chance of self-damage
                    self_damage = random.randint(1, 4)
                    chaos_unit['hp'] = max(1, chaos_unit['hp'] - self_damage)
                    battle_scene.battle_message += f" In its frenzy, it takes {self_damage} damage!"
                
                # Check if the player unit was defeated
                if nearest_player_unit['hp'] <= 0:
                    battle_scene._handle_player_unit_defeat(chaos_unit, nearest_player_unit)
                    battle_scene.battle_message = f"Enemy {chaos_unit['name']} slaughters {nearest_player_unit['name']} in a blood frenzy! Skulls for the Skull Throne!"
            else:
                # Move aggressively towards nearest unit - can move up to 2 cells if possible
                moved = False
                current_x, current_y = chaos_unit['position']
                target_x, target_y = nearest_player_unit['position']
                
                # Determine primary direction to move
                dx = 1 if target_x > current_x else -1 if target_x < current_x else 0
                dy = 1 if target_y > current_y else -1 if target_y < current_y else 0
                
                # Try primary movement direction (possibly moving 2 cells)
                if abs(target_x - current_x) > abs(target_y - current_y):
                    # Try horizontal movement first (2 cells if possible)
                    if battle_scene._try_move_enemy_unit(chaos_unit, current_x + dx * 2, current_y):
                        moved = True
                        battle_scene.battle_message = f"Enemy {chaos_unit['name']} charges furiously toward {nearest_player_unit['name']}!"
                    elif battle_scene._try_move_enemy_unit(chaos_unit, current_x + dx, current_y):
                        moved = True
                        battle_scene.battle_message = f"Enemy {chaos_unit['name']} advances toward {nearest_player_unit['name']}!"
                    elif battle_scene._try_move_enemy_unit(chaos_unit, current_x, current_y + dy):
                        moved = True
                        battle_scene.battle_message = f"Enemy {chaos_unit['name']} moves toward {nearest_player_unit['name']}!"
                else:
                    # Try vertical movement first (2 cells if possible)
                    if battle_scene._try_move_enemy_unit(chaos_unit, current_x, current_y + dy * 2):
                        moved = True
                        battle_scene.battle_message = f"Enemy {chaos_unit['name']} charges furiously toward {nearest_player_unit['name']}!"
                    elif battle_scene._try_move_enemy_unit(chaos_unit, current_x, current_y + dy):
                        moved = True
                        battle_scene.battle_message = f"Enemy {chaos_unit['name']} advances toward {nearest_player_unit['name']}!"
                    elif battle_scene._try_move_enemy_unit(chaos_unit, current_x + dx, current_y):
                        moved = True
                        battle_scene.battle_message = f"Enemy {chaos_unit['name']} moves toward {nearest_player_unit['name']}!"
                
                if not moved:
                    battle_scene.battle_message = f"Enemy {chaos_unit['name']} rages but can't move closer!"
            continue
            
        # Standard behavior (non-berserker, non-warp)
        if nearest_distance <= chaos_unit['attack_range']:
            # Calculate and apply damage
            chaos_damage = max(7, chaos_unit['attack'] - nearest_player_unit['defense'] // 3) + random.randint(-2, 5)
            nearest_player_unit['hp'] = max(0, nearest_player_unit['hp'] - chaos_damage)
            
            # Choose random chaos battle cry
            battle_cries = [
                f"Enemy {chaos_unit['name']} strikes {nearest_player_unit['name']} for {chaos_damage} damage!",
                f"Enemy {chaos_unit['name']} attacks {nearest_player_unit['name']} for {chaos_damage} damage in the name of Chaos!",
                f"Enemy {chaos_unit['name']} brings pain to {nearest_player_unit['name']} for {chaos_damage} damage!"
            ]
            battle_scene.battle_message = random.choice(battle_cries)
            
            # Check if the player unit was defeated
            if nearest_player_unit['hp'] <= 0:
                battle_scene._handle_player_unit_defeat(chaos_unit, nearest_player_unit)
                battle_scene.battle_message = f"Enemy {chaos_unit['name']} destroys {nearest_player_unit['name']}!"
        else:
            # Regular movement - prefer getting into attack range
            current_x, current_y = chaos_unit['position']
            target_x, target_y = nearest_player_unit['position']
            
            # Determine direction to move
            dx = 1 if target_x > current_x else -1 if target_x < current_x else 0
            dy = 1 if target_y > current_y else -1 if target_y < current_y else 0
            
            # Try to move
            if battle_scene._try_move_enemy_unit(chaos_unit, current_x + dx, current_y + dy) or \
               battle_scene._try_move_enemy_unit(chaos_unit, current_x + dx, current_y) or \
               battle_scene._try_move_enemy_unit(chaos_unit, current_x, current_y + dy):
                battle_scene.battle_message = f"Enemy {chaos_unit['name']} advances toward {nearest_player_unit['name']}." 