from game.ai.ork_ai import run_ork_ai
from game.ai.tau_ai import run_tau_ai
from game.ai.chaos_ai import run_chaos_ai
from game.ai.default_ai import run_default_ai

class AIManager:
    """Manager class to handle access to all faction-specific AI implementations"""
    
    def __init__(self):
        # Map of faction names to their AI handler functions
        self.faction_ai_map = {
            "orks": run_ork_ai,
            "tau": run_tau_ai,
            "chaos": run_chaos_ai,
            # Add more factions as they're implemented
        }
        
        # Default AI for factions without specific implementation
        self.default_ai = run_default_ai
    
    def process_faction_ai(self, battle_scene, faction_name, units):
        """Process AI for a specific faction's units"""
        # Convert faction name to lowercase for case-insensitive matching
        faction_key = faction_name.lower()
        
        # Get the appropriate AI function for this faction or use default
        ai_function = self.faction_ai_map.get(faction_key, self.default_ai)
        
        # Run the AI function
        ai_function(battle_scene, units)
    
    def get_faction_names(self):
        """Return a list of all supported faction names"""
        return list(self.faction_ai_map.keys())
    
    def register_faction_ai(self, faction_name, ai_function):
        """Register a new faction AI implementation (useful for mods)"""
        faction_key = faction_name.lower()
        self.faction_ai_map[faction_key] = ai_function 